"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Deployment = void 0;
const crypto = require("crypto");
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("./apigateway.generated");
const restapi_1 = require("./restapi");
/**
 * (experimental) A Deployment of a REST API.
 *
 * An immutable representation of a RestApi resource that can be called by users
 * using Stages. A deployment must be associated with a Stage for it to be
 * callable over the Internet.
 *
 * Normally, you don't need to define deployments manually. The RestApi
 * construct manages a Deployment resource that represents the latest model. It
 * can be accessed through `restApi.latestDeployment` (unless `deploy: false` is
 * set when defining the `RestApi`).
 *
 * If you manually define this resource, you will need to know that since
 * deployments are immutable, as long as the resource's logical ID doesn't
 * change, the deployment will represent the snapshot in time in which the
 * resource was created. This means that if you modify the RestApi model (i.e.
 * add methods or resources), these changes will not be reflected unless a new
 * deployment resource is created.
 *
 * To achieve this behavior, the method `addToLogicalId(data)` can be used to
 * augment the logical ID generated for the deployment resource such that it
 * will include arbitrary data. This is done automatically for the
 * `restApi.latestDeployment` deployment.
 *
 * Furthermore, since a deployment does not reference any of the REST API
 * resources and methods, CloudFormation will likely provision it before these
 * resources are created, which means that it will represent a "half-baked"
 * model. Use the `node.addDependency(dep)` method to circumvent that. This is done
 * automatically for the `restApi.latestDeployment` deployment.
 *
 * @experimental
 */
class Deployment extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.resource = new LatestDeploymentResource(this, 'Resource', {
            description: props.description,
            restApi: props.api,
        });
        if (props.retainDeployments) {
            this.resource.applyRemovalPolicy(core_1.RemovalPolicy.RETAIN);
        }
        this.api = props.api;
        this.deploymentId = core_1.Lazy.stringValue({ produce: () => this.resource.ref });
        if (props.api instanceof restapi_1.RestApiBase) {
            props.api._attachDeployment(this);
        }
    }
    /**
     * (experimental) Adds a component to the hash that determines this Deployment resource's logical ID.
     *
     * This should be called by constructs of the API Gateway model that want to
     * invalidate the deployment when their settings change. The component will
     * be resolve()ed during synthesis so tokens are welcome.
     *
     * @experimental
     */
    addToLogicalId(data) {
        this.resource.addToLogicalId(data);
    }
    /**
     * Quoting from CloudFormation's docs:
     *
     *   If you create an AWS::ApiGateway::RestApi resource and its methods (using
     *   AWS::ApiGateway::Method) in the same template as your deployment, the
     *   deployment must depend on the RestApi's methods. To create a dependency,
     *   add a DependsOn attribute to the deployment. If you don't, AWS
     *   CloudFormation creates the deployment right after it creates the RestApi
     *   resource that doesn't contain any methods, and AWS CloudFormation
     *   encounters the following error: The REST API doesn't contain any methods.
     *
     * @param method The method to add as a dependency of the deployment
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-deployment.html
     * @see https://github.com/aws/aws-cdk/pull/6165
     * @internal
     */
    _addMethodDependency(method) {
        // adding a dependency between the constructs using `node.addDependency()`
        // will create additional dependencies between `AWS::ApiGateway::Deployment`
        // and the `AWS::Lambda::Permission` resources (children under Method),
        // causing cyclic dependency errors. Hence, falling back to declaring
        // dependencies between the underlying CfnResources.
        this.node.addDependency(method.node.defaultChild);
    }
}
exports.Deployment = Deployment;
class LatestDeploymentResource extends apigateway_generated_1.CfnDeployment {
    constructor(scope, id, props) {
        super(scope, id, {
            description: props.description,
            restApiId: props.restApi.restApiId,
        });
        this.hashComponents = new Array();
        this.api = props.restApi;
        this.originalLogicalId = this.stack.getLogicalId(this);
        this.overrideLogicalId(core_1.Lazy.stringValue({ produce: () => this.calculateLogicalId() }));
    }
    /**
     * Allows adding arbitrary data to the hashed logical ID of this deployment.
     * This can be used to couple the deployment to the API Gateway model.
     */
    addToLogicalId(data) {
        // if the construct is locked, it means we are already synthesizing and then
        // we can't modify the hash because we might have already calculated it.
        if (this.node.locked) {
            throw new Error('Cannot modify the logical ID when the construct is locked');
        }
        this.hashComponents.push(data);
    }
    calculateLogicalId() {
        const hash = [...this.hashComponents];
        if (this.api instanceof restapi_1.RestApi || this.api instanceof restapi_1.SpecRestApi) { // Ignore IRestApi that are imported
            // Add CfnRestApi to the logical id so a new deployment is triggered when any of its properties change.
            const cfnRestApiCF = this.api.node.defaultChild._toCloudFormation();
            hash.push(this.stack.resolve(cfnRestApiCF));
        }
        let lid = this.originalLogicalId;
        // if hash components were added to the deployment, we use them to calculate
        // a logical ID for the deployment resource.
        if (hash.length > 0) {
            const md5 = crypto.createHash('md5');
            hash.map(x => this.stack.resolve(x)).forEach(c => md5.update(JSON.stringify(c)));
            lid += md5.digest('hex');
        }
        return lid;
    }
}
//# sourceMappingURL=data:application/json;base64,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