"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RateLimitedApiKey = exports.ApiKey = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("./apigateway.generated");
const usage_plan_1 = require("./usage-plan");
/**
 * Base implementation that is common to the various implementations of IApiKey
 */
class ApiKeyBase extends core_1.Resource {
    /**
     * (experimental) Permits the IAM principal all read operations through this key.
     *
     * @param grantee The principal to grant access to.
     * @experimental
     */
    grantRead(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: readPermissions,
            resourceArns: [this.keyArn],
        });
    }
    /**
     * (experimental) Permits the IAM principal all write operations through this key.
     *
     * @param grantee The principal to grant access to.
     * @experimental
     */
    grantWrite(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: writePermissions,
            resourceArns: [this.keyArn],
        });
    }
    /**
     * (experimental) Permits the IAM principal all read and write operations through this key.
     *
     * @param grantee The principal to grant access to.
     * @experimental
     */
    grantReadWrite(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: [...readPermissions, ...writePermissions],
            resourceArns: [this.keyArn],
        });
    }
}
/**
 * (experimental) An API Gateway ApiKey.
 *
 * An ApiKey can be distributed to API clients that are executing requests
 * for Method resources that require an Api Key.
 *
 * @experimental
 */
class ApiKey extends ApiKeyBase {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.apiKeyName,
        });
        const resource = new apigateway_generated_1.CfnApiKey(this, 'Resource', {
            customerId: props.customerId,
            description: props.description,
            enabled: props.enabled || true,
            generateDistinctId: props.generateDistinctId,
            name: this.physicalName,
            stageKeys: this.renderStageKeys(props.resources),
            value: props.value,
        });
        this.keyId = resource.ref;
        this.keyArn = core_1.Stack.of(this).formatArn({
            service: 'apigateway',
            account: '',
            resource: '/apikeys',
            sep: '/',
            resourceName: this.keyId,
        });
    }
    /**
     * (experimental) Import an ApiKey by its Id.
     *
     * @experimental
     */
    static fromApiKeyId(scope, id, apiKeyId) {
        class Import extends ApiKeyBase {
            constructor() {
                super(...arguments);
                this.keyId = apiKeyId;
                this.keyArn = core_1.Stack.of(this).formatArn({
                    service: 'apigateway',
                    account: '',
                    resource: '/apikeys',
                    sep: '/',
                    resourceName: apiKeyId,
                });
            }
        }
        return new Import(scope, id);
    }
    renderStageKeys(resources) {
        if (!resources) {
            return undefined;
        }
        return resources.map((resource) => {
            const restApi = resource;
            const restApiId = restApi.restApiId;
            const stageName = restApi.deploymentStage.stageName.toString();
            return { restApiId, stageName };
        });
    }
}
exports.ApiKey = ApiKey;
/**
 * (experimental) An API Gateway ApiKey, for which a rate limiting configuration can be specified.
 *
 * @experimental
 * @resource AWS::ApiGateway::ApiKey
 */
class RateLimitedApiKey extends ApiKeyBase {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.apiKeyName,
        });
        const resource = new ApiKey(this, 'Resource', props);
        if (props.apiStages || props.quota || props.throttle) {
            new usage_plan_1.UsagePlan(this, 'UsagePlanResource', {
                apiKey: resource,
                apiStages: props.apiStages,
                quota: props.quota,
                throttle: props.throttle,
            });
        }
        this.keyId = resource.keyId;
        this.keyArn = resource.keyArn;
    }
}
exports.RateLimitedApiKey = RateLimitedApiKey;
const readPermissions = [
    'apigateway:GET',
];
const writePermissions = [
    'apigateway:POST',
    'apigateway:PUT',
    'apigateway:PATCH',
    'apigateway:DELETE',
];
//# sourceMappingURL=data:application/json;base64,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