"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetApiDefinition = exports.InlineApiDefinition = exports.S3ApiDefinition = exports.ApiDefinition = void 0;
const s3_assets = require("../../aws-s3-assets"); // Automatically re-written from '@aws-cdk/aws-s3-assets'
/**
 * (experimental) Represents an OpenAPI definition asset.
 *
 * @experimental
 */
class ApiDefinition {
    /**
     * (experimental) Creates an API definition from a specification file in an S3 bucket.
     *
     * @experimental
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3ApiDefinition(bucket, key, objectVersion);
    }
    /**
     * (experimental) Create an API definition from an inline object.
     *
     * The inline object must follow the
     * schema of OpenAPI 2.0 or OpenAPI 3.0
     *
     * @experimental
     * @example
     *
     *   ApiDefinition.fromInline({
     *     openapi: '3.0.2',
     *     paths: {
     *       '/pets': {
     *         get: {
     *           'responses': {
     *             200: {
     *               content: {
     *                 'application/json': {
     *                   schema: {
     *                     $ref: '#/components/schemas/Empty',
     *                   },
     *                 },
     *               },
     *             },
     *           },
     *           'x-amazon-apigateway-integration': {
     *             responses: {
     *               default: {
     *                 statusCode: '200',
     *               },
     *             },
     *             requestTemplates: {
     *               'application/json': '{"statusCode": 200}',
     *             },
     *             passthroughBehavior: 'when_no_match',
     *             type: 'mock',
     *           },
     *         },
     *       },
     *     },
     *     components: {
     *       schemas: {
     *         Empty: {
     *           title: 'Empty Schema',
     *           type: 'object',
     *         },
     *       },
     *     },
     *   });
     */
    static fromInline(definition) {
        return new InlineApiDefinition(definition);
    }
    /**
     * (experimental) Loads the API specification from a local disk asset.
     *
     * @experimental
     */
    static fromAsset(file, options) {
        return new AssetApiDefinition(file, options);
    }
}
exports.ApiDefinition = ApiDefinition;
/**
 * (experimental) OpenAPI specification from an S3 archive.
 *
 * @experimental
 */
class S3ApiDefinition extends ApiDefinition {
    /**
     * @experimental
     */
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    /**
     * (experimental) Called when the specification is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @experimental
     */
    bind(_scope) {
        return {
            s3Location: {
                bucket: this.bucketName,
                key: this.key,
                version: this.objectVersion,
            },
        };
    }
}
exports.S3ApiDefinition = S3ApiDefinition;
/**
 * (experimental) OpenAPI specification from an inline JSON object.
 *
 * @experimental
 */
class InlineApiDefinition extends ApiDefinition {
    /**
     * @experimental
     */
    constructor(definition) {
        super();
        this.definition = definition;
        if (typeof (definition) !== 'object') {
            throw new Error('definition should be of type object');
        }
        if (Object.keys(definition).length === 0) {
            throw new Error('JSON definition cannot be empty');
        }
    }
    /**
     * (experimental) Called when the specification is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @experimental
     */
    bind(_scope) {
        return {
            inlineDefinition: this.definition,
        };
    }
}
exports.InlineApiDefinition = InlineApiDefinition;
/**
 * (experimental) OpenAPI specification from a local file.
 *
 * @experimental
 */
class AssetApiDefinition extends ApiDefinition {
    /**
     * @experimental
     */
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
    }
    /**
     * (experimental) Called when the specification is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @experimental
     */
    bind(scope) {
        // If the same AssetAPIDefinition is used multiple times, retain only the first instantiation.
        if (this.asset === undefined) {
            this.asset = new s3_assets.Asset(scope, 'APIDefinition', {
                path: this.path,
                ...this.options,
            });
        }
        if (this.asset.isZipArchive) {
            throw new Error(`Asset cannot be a .zip file or a directory (${this.path})`);
        }
        return {
            s3Location: {
                bucket: this.asset.s3BucketName,
                key: this.asset.s3ObjectKey,
            },
        };
    }
}
exports.AssetApiDefinition = AssetApiDefinition;
//# sourceMappingURL=data:application/json;base64,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