# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['dataclass_mapper']

package_data = \
{'': ['*']}

extras_require = \
{'pydantic': ['pydantic>=1.9.0,<2.0.0']}

setup_kwargs = {
    'name': 'dataclass-mapper',
    'version': '1.0.2',
    'description': 'Autogenerate mappings between dataclasses',
    'long_description': 'dataclass-mapper\n================\n\n|pypi| |support| |licence| |readthedocs| |build| |coverage|\n\n.. |pypi| image:: https://img.shields.io/pypi/v/dataclass-mapper.svg?style=flat-square\n    :target: https://pypi.org/project/dataclass-mapper/\n    :alt: pypi version\n\n.. |support| image:: https://img.shields.io/pypi/pyversions/dataclass-mapper.svg?style=flat-square\n    :target: https://pypi.org/project/dataclass-mapper/\n    :alt: supported Python version\n\n.. |build| image:: https://github.com/dataclass-mapper/dataclass-mapper/actions/workflows/test.yml/badge.svg\n    :target: https://github.com/dataclass-mapper/dataclass-mapper/actions\n    :alt: build status\n\n.. |coverage| image:: https://codecov.io/gh/dataclass-mapper/dataclass-mapper/branch/main/graphs/badge.svg?branch=main\n    :target: https://codecov.io/gh/dataclass-mapper/dataclass-mapper?branch=main\n    :alt: Code coverage\n\n.. |licence| image:: https://img.shields.io/pypi/l/dataclass-mapper.svg?style=flat-square\n    :target: https://pypi.org/project/dataclass-mapper/\n    :alt: licence\n\n.. |readthedocs| image:: https://img.shields.io/readthedocs/dataclass-mapper/latest.svg?style=flat-square&label=Read%20the%20Docs\n   :alt: Read the documentation at https://dataclass-mapper.readthedocs.io/en/latest/\n   :target: https://dataclass-mapper.readthedocs.io/en/latest/\n\nWriting mapper methods between two similar dataclasses is boring, need to be actively maintained and are error-prone.\nMuch better to let a library auto-generate them for you.\n\nThis library makes it easy to autogenerate mappers, makes sure that the types between source and target class match, and that all fields of the target class are actually mapped to.\nMost of those checks are already done at class definition time, not when the mappings are run.\nIt supports Python\'s dataclasses and also Pydantic models, and can also map between those two.\n\nInstallation\n------------\n\n``dataclass-mapper`` can be installed using:\n\n.. code-block:: bash\n\n    pip install dataclass-mapper\n    # or for Pydantic support\n    pip install dataclass-mapper[pydantic]\n\nSmall Example\n-------------\n\nWe have the following target data structure, a class called ``Person``.\n\n.. code-block:: python\n\n    from dataclasses import dataclass\n\n    @dataclass\n    class Person:\n        first_name: str\n        second_name: str\n        age: int\n\nWe want to have a mapper from the source data structure, a class called ``ContactInfo``.\nNotice that the attribute ``second_name`` of `Person` is called ``surname`` in `ContactInfo`.\nOther than that, all the attribute names are the same.\n\nInstead of writing a mapper `to_Person` by hand:\n\n.. code-block:: python\n\n    @dataclass\n    class ContactInfo:\n        first_name: str\n        surname: str\n        age: int\n\n        def to_Person(self) -> Person:\n            return Person(\n                first_name=self.first_name,\n                second_name=self.surname,\n                age=self.age,\n            )\n\n    person = some_contact.to_Person()\n\nyou can let the mapper autogenerate with:\n\n.. code-block:: python\n\n    from dataclass_mapper import map_to, mapper\n\n    @mapper(Person, {"second_name": "surname"})\n    @dataclass\n    class ContactInfo:\n        first_name: str\n        surname: str\n        age: int\n\n    person = map_to(some_contact, Person)\n\nThe ``dataclass-mapper`` library autogenerated some a mapper, that can be used with the ``map_to`` function.\nAll we had to specify was the name of the target class, and optionally specify which fields map to which other fields.\nNotice that we only had to specify that the ``second_name`` field has to be mapped to ``surname``,\nall other fields were mapped automatically because the field names didn\'t change.\n\nAnd the ``dataclass-mapper`` library will perform a lot of checks around this mapping.\nIt will check if the data types match, if some fields would be left uninitialized, etc.\n\nFeatures\n--------\n\nThe current version has support for:\n\n* Python\'s ``dataclass``\n* ``pydantic`` classes\n* Checks if all target fields are actually initialized.\n  Raises a ``ValueError`` at class definition time when the type is different.\n* Checks if the type on the target field is the same as the source field.\n  Raises a ``TypeError`` at class definition time when the type is different.\n* Recursive dataclasses\n* ``USE_DEFAULT`` for values that you don\'t wanna set but have a default value/factory.\n* ``Optional`` types (mapping from an non-optional to an optional field, or to an optional field with default values/fields).\n  Raises a ``TypeError`` at class definition time when an optional type is mapped to a non-optional type.\n* ``List`` types\n* Mapper in both direction with ``mapper`` and ``mapper_from``.\n* Assign Values with lambdas (e.g. ``{"x": lambda: 42}``)\n* Custom mapping computations with with lambdas (e.g. ``{"x": lambda self: self.x + 1}``)\n\nStill missing features:\n\n* ``Union`` types\n* ``Dict`` types\n* Aliases in `pydantic` classes\n* Checking if all source attributes were used\n* SQLAlchemy ORM / attr\n\nLicense\n-------\n\nThe project is released under the `MIT license <https://github.com/dataclass-mapper/dataclass-mapper/blob/main/LICENSE.md>`_.\n',
    'author': 'Jakob Kogler',
    'author_email': 'jakob.kogler@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://dataclass-mapper.readthedocs.io',
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
