"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Monitoring = void 0;
const cw = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const cdk_watchful_1 = require("cdk-watchful");
const web_canary_1 = require("./web-canary");
/**
 * Construct hub monitoring.
 *
 * This construct uses cdk-watchful to automatically generate a dashboard and a
 * set of standard alarms for common resources.
 *
 * Components should use the APIs of this module to add watchful monitors,
 * and add canaries and alarms as needed.
 */
class Monitoring extends core_1.Construct {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id);
        this.alarmActions = props.alarmActions;
        this.watchful = new cdk_watchful_1.Watchful(this, 'Watchful', {
            // alarms that come from watchful are all considered normal severity
            alarmActionArns: ((_a = this.alarmActions) === null || _a === void 0 ? void 0 : _a.normalSeverity) ? [this.alarmActions.normalSeverity] : [],
            alarmActions: ((_b = this.alarmActions) === null || _b === void 0 ? void 0 : _b.normalSeverityAction) ? [this.alarmActions.normalSeverityAction] : [],
        });
        this.highSeverityDashboard = new cw.Dashboard(this, 'HighSeverityDashboard');
    }
    /**
     * Adds a high-severity alarm. If this alarm goes off, the action specified in `highSeverityAlarmActionArn`
     * @param alarm
     */
    addHighSeverityAlarm(title, alarm) {
        var _a, _b;
        const highSeverityActionArn = (_a = this.alarmActions) === null || _a === void 0 ? void 0 : _a.highSeverity;
        if (highSeverityActionArn) {
            alarm.addAlarmAction({
                bind: () => ({ alarmActionArn: highSeverityActionArn }),
            });
        }
        const highSeverityAction = (_b = this.alarmActions) === null || _b === void 0 ? void 0 : _b.highSeverityAction;
        if (highSeverityAction) {
            alarm.addAlarmAction(highSeverityAction);
        }
        this.highSeverityDashboard.addWidgets(new cw.AlarmWidget({
            alarm,
            title,
            width: 24,
        }));
    }
    /**
     * Adds a canary that pings a certain URL and raises an alarm in case the URL
     * responds with an error over 80% of the times.
     *
     * Canary failures are always considered high-severity (at this point).
     *
     * @param name The name of the canary
     * @param url The URL to ping
     */
    addWebCanary(name, url) {
        const canary = new web_canary_1.WebCanary(this, `WebCanary${name.replace(/[^A-Z0-9]/ig, '')}`, {
            url,
            displayName: name,
        });
        this.addHighSeverityAlarm(`${name} Canary`, canary.alarm);
    }
}
exports.Monitoring = Monitoring;
//# sourceMappingURL=data:application/json;base64,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