"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const Environments_1 = require("aws-embedded-metrics/lib/environment/Environments");
const semver_1 = require("semver");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.environmentOverride = Environments_1.default.Lambda;
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
async function handler(event, _context) {
    var _a;
    console.log('Event:', JSON.stringify(event, null, 2));
    const indexedPackages = new Map();
    const packageNames = new Set();
    const packageMajorVersions = new Set();
    const perLanguage = new Map();
    /**
     * Records the status of a particular package, package major version, package
     * version, and package version submodule in the per-language state storage.
     * Whenever a new entry is added, a `MISSING` entry is automatically inserted
     * for the other languages (unless another entry already exists).
     *
     * If a submodule is provided, only that submodule's availability is updated.
     */
    function recordPerLanguage(language, status, pkgName, pkgMajor, pkgVersion, submodule) {
        for (const lang of language_1.DocumentationLanguage.ALL) {
            doRecordPerLanguage(perLanguage, lang, 
            // If the language is NOT the registered one, then we insert "MISSING".
            lang === language ? status : "Missing" /* MISSING */, pkgName, pkgMajor, pkgVersion, submodule);
        }
    }
    const bucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    for await (const key of relevantObjectKeys(bucket)) {
        const [, name, version] = constants.STORAGE_KEY_FORMAT_REGEX.exec(key);
        packageNames.add(name);
        const majorVersion = `${name}@${new semver_1.SemVer(version).major}`;
        packageMajorVersions.add(majorVersion);
        const fullName = `${name}@${version}`;
        // Ensure the package is fully registered for per-language status, even if no doc exists yet.
        for (const language of language_1.DocumentationLanguage.ALL) {
            recordPerLanguage(language, "Missing" /* MISSING */, name, majorVersion, fullName);
        }
        if (!indexedPackages.has(fullName)) {
            indexedPackages.set(fullName, {});
        }
        const status = indexedPackages.get(fullName);
        if (key.endsWith(constants.METADATA_KEY_SUFFIX)) {
            status.metadataPresent = true;
        }
        else if (key.endsWith(constants.PACKAGE_KEY_SUFFIX)) {
            status.tarballPresent = true;
        }
        else if (key.endsWith(constants.ASSEMBLY_KEY_SUFFIX)) {
            status.assemblyPresent = true;
        }
        else {
            let identified = false;
            for (const language of language_1.DocumentationLanguage.ALL) {
                const match = submoduleKeyRegexp(language).exec(key);
                if (match != null) {
                    const [, submodule, isUnsupported] = match;
                    if (status.submodules == null) {
                        status.submodules = new Set();
                    }
                    status.submodules.add(`${fullName}.${submodule}`);
                    recordPerLanguage(language, isUnsupported ? "Unsupported" /* UNSUPPORTED */ : "Supported" /* SUPPORTED */, name, majorVersion, fullName, submodule);
                    identified = true;
                }
                else if (key.endsWith(constants.docsKeySuffix(language))) {
                    recordPerLanguage(language, "Supported" /* SUPPORTED */, name, majorVersion, fullName);
                    identified = true;
                }
                else if (key.endsWith(constants.docsKeySuffix(language) + constants.NOT_SUPPORTED_SUFFIX)) {
                    recordPerLanguage(language, "Supported" /* SUPPORTED */, name, majorVersion, fullName);
                    identified = true;
                }
            }
            if (!identified) {
                status.unknownObjects = (_a = status.unknownObjects) !== null && _a !== void 0 ? _a : [];
                status.unknownObjects.push(key);
            }
        }
    }
    await aws_embedded_metrics_1.metricScope((metrics) => () => {
        var _a, _b, _c;
        // Clear out default dimensions as we don't need those. See https://github.com/awslabs/aws-embedded-metrics-node/issues/73.
        metrics.setDimensions();
        const missingMetadata = new Array();
        const missingAssembly = new Array();
        const missingTarball = new Array();
        const unknownObjects = new Array();
        const submodules = new Array();
        for (const [name, status] of indexedPackages.entries()) {
            if (!status.metadataPresent) {
                missingMetadata.push(name);
            }
            if (!status.assemblyPresent) {
                missingAssembly.push(name);
            }
            if (!status.tarballPresent) {
                missingTarball.push(name);
            }
            if ((_b = (_a = status.unknownObjects) === null || _a === void 0 ? void 0 : _a.length) !== null && _b !== void 0 ? _b : 0 > 0) {
                unknownObjects.push(...status.unknownObjects);
            }
            for (const submodule of (_c = status.submodules) !== null && _c !== void 0 ? _c : []) {
                submodules.push(submodule);
            }
        }
        metrics.setProperty('detail', { missingMetadata, missingAssembly, missingTarball, unknownObjects });
        metrics.putMetric("MissingPackageMetadataCount" /* MISSING_METADATA_COUNT */, missingMetadata.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingAssemblyCount" /* MISSING_ASSEMBLY_COUNT */, missingAssembly.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingPackageTarballCount" /* MISSING_TARBALL_COUNT */, missingTarball.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageCount" /* PACKAGE_COUNT */, packageNames.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageMajorVersionCount" /* PACKAGE_MAJOR_COUNT */, packageMajorVersions.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageVersionCount" /* PACKAGE_VERSION_COUNT */, indexedPackages.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("SubmoduleCount" /* SUBMODULE_COUNT */, submodules.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("UnknownObjectCount" /* UNKNOWN_OBJECT_COUNT */, unknownObjects.length, aws_embedded_metrics_1.Unit.Count);
    })();
    for (const [language, data] of perLanguage.entries()) {
        await aws_embedded_metrics_1.metricScope((metrics) => () => {
            metrics.setDimensions({ [constants_1.LANGUAGE_DIMENSION]: language.toString() });
            for (const forStatus of ["Supported" /* SUPPORTED */, "Unsupported" /* UNSUPPORTED */, "Missing" /* MISSING */]) {
                for (const [key, statuses] of Object.entries(data)) {
                    let filtered = Array.from(statuses.entries()).filter(([, status]) => forStatus === status);
                    let metricName = METRIC_NAME_BY_STATUS_AND_GRAIN[forStatus][key];
                    // List out selected packages for posterity (and troubleshooting)
                    console.log(`${forStatus} ${key} for ${language}: ${filtered.map(([name]) => name).join(', ')}`);
                    metrics.putMetric(metricName, filtered.length, aws_embedded_metrics_1.Unit.Count);
                }
            }
        })();
    }
}
exports.handler = handler;
async function* relevantObjectKeys(bucket) {
    var _a;
    const request = {
        Bucket: bucket,
        Prefix: constants.STORAGE_KEY_PREFIX,
    };
    do {
        const response = await aws.s3().listObjectsV2(request).promise();
        for (const { Key } of (_a = response.Contents) !== null && _a !== void 0 ? _a : []) {
            if (Key == null) {
                continue;
            }
            yield Key;
        }
        request.ContinuationToken = response.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
/**
 * This function obtains a regular expression with a capture group that allows
 * determining the submodule name from a submodule documentation key, and
 * another to determine whether the object is an "unsupported beacon" or not.
 */
function submoduleKeyRegexp(language) {
    // We use a placeholder to be able to insert the capture group once we have
    // fully quoted the key prefix for Regex safety.
    const placeholder = '<SUBMODULENAME>';
    // We obtain the standard key prefix.
    const keyPrefix = constants.docsKeySuffix(language, placeholder);
    // Finally, assemble the regular expression with the capture group.
    return new RegExp(`.*${reQuote(keyPrefix).replace(placeholder, '(.+)')}(${reQuote(constants.NOT_SUPPORTED_SUFFIX)})?$`);
    /**
     * Escapes all "speacial meaning" characters in a string, so it can be used as
     * part of a regular expression.
     */
    function reQuote(str) {
        return str.replace(/([+*.()?$[\]])/g, '\\$1');
    }
}
const METRIC_NAME_BY_STATUS_AND_GRAIN = {
    ["Missing" /* MISSING */]: {
        ["packages" /* PACKAGES */]: "MissingPackageCount" /* PER_LANGUAGE_MISSING_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "MissingMajorVersionCount" /* PER_LANGUAGE_MISSING_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "MissingPackageVersionCount" /* PER_LANGUAGE_MISSING_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "MissingSubmoduleCount" /* PER_LANGUAGE_MISSING_SUBMODULES */,
    },
    ["Unsupported" /* UNSUPPORTED */]: {
        ["packages" /* PACKAGES */]: "UnsupportedPackageCount" /* PER_LANGUAGE_UNSUPPORTED_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "UnsupportedMajorVersionCount" /* PER_LANGUAGE_UNSUPPORTED_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "UnsupportedPackageVersionCount" /* PER_LANGUAGE_UNSUPPORTED_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "UnsupportedSubmoduleCount" /* PER_LANGUAGE_UNSUPPORTED_SUBMODULES */,
    },
    ["Supported" /* SUPPORTED */]: {
        ["packages" /* PACKAGES */]: "SupportedPackageCount" /* PER_LANGUAGE_SUPPORTED_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "SupportedMajorVersionCount" /* PER_LANGUAGE_SUPPORTED_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "SupportedPackageVersionCount" /* PER_LANGUAGE_SUPPORTED_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "SupportedSubmoduleCount" /* PER_LANGUAGE_SUPPORTED_SUBMODULES */,
    },
};
/**
 * Registers the information for the provided language. A "MISSING" status
 * will be ignored if another status was already registered for the same
 * entity. An "UNSUPPORTED" status will be ignored if a "SUPPORTED" status
 * was already registered for the same entity.
 *
 * If a submodule is provided, only that submodule's availability is updated.
 */
function doRecordPerLanguage(perLanguage, language, status, pkgName, pkgMajor, pkgVersion, submodule) {
    if (!perLanguage.has(language)) {
        perLanguage.set(language, {
            ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: new Map(),
            ["packages" /* PACKAGES */]: new Map(),
            ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: new Map(),
            ["package versions" /* PACKAGE_VERSIONS */]: new Map(),
        });
    }
    const data = perLanguage.get(language);
    // If there is a submodule, only update the submodule domain.
    const outputDomains = submodule
        ? [
            [data["package version submodules" /* PACKAGE_VERSION_SUBMODULES */], `${pkgVersion}.${submodule}`],
        ]
        : [
            [data["package major versions" /* PACKAGE_MAJOR_VERSIONS */], pkgMajor],
            [data["package versions" /* PACKAGE_VERSIONS */], pkgVersion],
            [data["packages" /* PACKAGES */], pkgName],
        ];
    for (const [map, name] of outputDomains) {
        switch (status) {
            case "Missing" /* MISSING */:
                // If we already have a status, don't override it with "MISSING".
                if (!map.has(name)) {
                    map.set(name, status);
                }
                break;
            case "Supported" /* SUPPORTED */:
                // If thr package is "supported", this always "wins"
                map.set(name, status);
                break;
            case "Unsupported" /* UNSUPPORTED */:
                // If we already have a status, only override with "UNSUPPORTED" if it was "MISSING".
                if (!map.has(name) || map.get(name) === "Missing" /* MISSING */) {
                    map.set(name, status);
                }
                break;
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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