"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const client_lambda_shared_1 = require("../../../backend/deny-list/client.lambda-shared");
const constants_1 = require("../../../backend/deny-list/constants");
const aws = require("../../../backend/shared/aws.lambda-shared");
const sample = {
    'foo/v1.2.3': {
        package: 'foo',
        version: '1.2.3',
        reason: 'bar',
    },
    'bar': {
        package: 'bar',
        reason: 'hello bar',
    },
};
beforeEach(() => {
    process.env[constants_1.ENV_DENY_LIST_BUCKET_NAME] = 'deny-list-bucket-name';
    process.env[constants_1.ENV_DENY_LIST_OBJECT_KEY] = 'deny-list.json';
    AWSMock.setSDKInstance(AWS);
});
afterEach(() => {
    delete process.env[constants_1.ENV_DENY_LIST_BUCKET_NAME];
    delete process.env[constants_1.ENV_DENY_LIST_OBJECT_KEY];
    AWSMock.restore();
    aws.reset();
});
test('s3 object not found error', async () => {
    AWSMock.mock('S3', 'getObject', (_, callback) => {
        const err = new Error('NoSuchKey');
        err.code = 'NoSuchKey';
        callback(err, null);
    });
    const client = await client_lambda_shared_1.DenyListClient.newClient();
    expect(client.lookup('foo', '1.2.3')).toBeUndefined();
});
test('s3 bucket not found error', async () => {
    AWSMock.mock('S3', 'getObject', (_, callback) => {
        const err = new Error('NoSuchBucket');
        err.code = 'NoSuchBucket';
        callback(err, null);
    });
    const client = await client_lambda_shared_1.DenyListClient.newClient();
    expect(client.lookup('foo', '1.2.3')).toBeUndefined();
});
test('empty file', async () => {
    AWSMock.mock('S3', 'getObject', (params, callback) => {
        expect(params.Bucket).toBe('deny-list-bucket-name');
        expect(params.Key).toBe('deny-list.json');
        callback(null, { Body: '' });
    });
    const client = await client_lambda_shared_1.DenyListClient.newClient();
    expect(client.lookup('foo', '1.2.3')).toBeUndefined();
});
test('json parsing error', async () => {
    AWSMock.mock('S3', 'getObject', (params, callback) => {
        expect(params.Bucket).toBe('deny-list-bucket-name');
        expect(params.Key).toBe('deny-list.json');
        callback(null, { Body: '09x{}' });
    });
    const expected = new Error('Unable to parse deny list file deny-list-bucket-name/deny-list.json: SyntaxError: Unexpected number in JSON at position 1');
    await expect(client_lambda_shared_1.DenyListClient.newClient()).rejects.toEqual(expected);
});
describe('lookup', () => {
    let client;
    beforeEach(async () => {
        AWSMock.mock('S3', 'getObject', (params, callback) => {
            expect(params.Bucket).toBe('deny-list-bucket-name');
            expect(params.Key).toBe('deny-list.json');
            callback(null, { Body: JSON.stringify(sample) });
        });
        client = await client_lambda_shared_1.DenyListClient.newClient();
    });
    test('match specific package + version', () => {
        expect(client.lookup('foo', '1.2.3')).toStrictEqual({
            package: 'foo',
            version: '1.2.3',
            reason: 'bar',
        });
    });
    test('match any version', () => {
        const expected = {
            package: 'bar',
            reason: 'hello bar',
        };
        expect(client.lookup('bar', '1.2.3')).toStrictEqual(expected);
        expect(client.lookup('bar', '4.4.4')).toStrictEqual(expected);
    });
    test('version does not match', () => {
        expect(client.lookup('foo', '4.4.4')).toBeUndefined();
    });
});
//# sourceMappingURL=data:application/json;base64,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