"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebCanary = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const core_1 = require("@aws-cdk/core");
const http_get_function_1 = require("./http-get-function");
/**
 * A canary that periodically sends an HTTP GET request to a specific URL.
 */
class WebCanary extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const url = props.url;
        const display = props.displayName;
        const ping = new http_get_function_1.HttpGetFunction(this, 'HttpGetFunction', {
            description: `HTTP GET ${url}: ${display}`,
            environment: { URL: url },
        });
        // invoke every 1min
        new events.Rule(this, 'Rule', {
            schedule: events.Schedule.rate(core_1.Duration.minutes(1)),
            targets: [new targets.LambdaFunction(ping)],
        });
        // total number of errors every 5 minutes
        const errors = ping.metricErrors({
            period: core_1.Duration.minutes(5),
            statistic: 'sum',
            label: `${url} Errors`,
        });
        // alarm if 4 or more pings have failed within a period of 5 minutes (80% failure rate)
        this.alarm = errors.createAlarm(this, 'Errors', {
            alarmDescription: `80% error rate for ${props.url} (${display})`,
            threshold: 4,
            evaluationPeriods: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
        });
    }
}
exports.WebCanary = WebCanary;
//# sourceMappingURL=data:application/json;base64,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