"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const child_process_1 = require("child_process");
const console = require("console");
const os = require("os");
const path = require("path");
const process = require("process");
const aws_sdk_1 = require("aws-sdk");
const fs = require("fs-extra");
const jsii_rosetta_1 = require("jsii-rosetta");
const transliterate_1 = require("jsii-rosetta/lib/commands/transliterate");
const clients = new Map();
const PACKAGE_KEY_REGEX = /^packages\/((?:@[^/]+\/)?[^/]+)\/v([^/]+)\/package.tgz$/;
/**
 * This function receives an S3 event, and for each record, proceeds to download
 * the `.jsii` assembly the event refers to, transliterates it to Python, then
 * uploads the resulting `.jsii.python` object to S3.
 *
 * @param event   an S3 event payload
 * @param context a Lambda execution context
 *
 * @returns nothing
 */
async function handler(event, context) {
    var _a, _b;
    console.error(JSON.stringify(event, null, 2));
    const created = new Array();
    for (const record of event.Records) {
        const [, packageName, packageVersion] = (_a = record.s3.object.key.match(PACKAGE_KEY_REGEX)) !== null && _a !== void 0 ? _a : [];
        if (packageName == null) {
            throw new Error(`Invalid object key: "${record.s3.object.key}". It was expected to match ${PACKAGE_KEY_REGEX}!`);
        }
        const client = (clients.has(record.awsRegion)
            ? clients
            : clients.set(record.awsRegion, new aws_sdk_1.S3({ region: record.awsRegion }))).get(record.awsRegion);
        const object = await client.getObject({
            Bucket: record.s3.bucket.name,
            Key: record.s3.object.key,
            VersionId: record.s3.object.versionId,
        }).promise();
        const workdir = await fs.mkdtemp(path.join(os.tmpdir(), 'workdir'));
        try {
            const tarball = path.join(workdir, `${packageName.replace('@', '').replace('/', '-')}-${packageVersion}.tgz`);
            await fs.writeFile(tarball, object.Body);
            await new Promise((ok, ko) => {
                // --ignore-scripts disables lifecycle hooks, in order to prevent execution of arbitrary code
                // --no-bin-links ensures npm does not insert anything in $PATH
                const npmInstall = child_process_1.spawn('npm', ['install', '--ignore-scripts', '--no-bin-links', '--no-save', tarball], {
                    cwd: workdir,
                    env: {
                        ...process.env,
                        HOME: os.tmpdir(),
                    },
                    stdio: ['ignore', 'inherit', 'inherit'],
                });
                npmInstall.once('error', ko);
                npmInstall.once('close', (code, signal) => {
                    if (code === 0) {
                        ok();
                    }
                    else {
                        ko(`"npm install" command ${code != null ? `exited with code ${code}` : `was terminated by signal ${signal}`}`);
                    }
                });
            });
            const packageDir = path.join(workdir, 'node_modules', packageName);
            await transliterate_1.transliterateAssembly([packageDir], [jsii_rosetta_1.TargetLanguage.PYTHON]);
            // Payload object key => packages/[<@scope>/]<name>/v<version>/package.tgz
            // Output object key  => packages/[<@scope>/]<name>/v<version>/assembly-python.json
            const key = record.s3.object.key.replace(/\/[^/]+$/, '/assembly-python.json');
            const response = await client.putObject({
                Bucket: record.s3.bucket.name,
                Key: key,
                Body: await fs.readFile(path.join(packageDir, '.jsii.python')),
                ContentType: 'text/json',
                Metadata: {
                    'Origin-Version-Id': (_b = record.s3.object.versionId) !== null && _b !== void 0 ? _b : 'N/A',
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise();
            created.push({
                bucket: record.s3.bucket.name,
                key,
                versionId: response.VersionId,
            });
        }
        finally {
            await fs.remove(workdir);
        }
    }
    return created;
}
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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