"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto_1 = require("crypto");
const events_1 = require("events");
const fs_1 = require("fs");
const path = require("path");
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const jsii_rosetta_1 = require("jsii-rosetta");
const transliterator_lambda_1 = require("../../../backend/transliterator/transliterator.lambda");
jest.mock('child_process');
jest.mock('jsii-rosetta/lib/commands/transliterate');
beforeEach((done) => {
    AWSMock.setSDKInstance(AWS);
    done();
});
afterEach((done) => {
    AWSMock.restore();
    jest.resetAllMocks();
    done();
});
test('basic usage', async () => {
    // GIVEN
    const packageName = 'package-name';
    const packageVersion = '1.2.3-dev.4';
    const payload = {
        Records: [{
                awsRegion: 'bemuda-triangle-1',
                eventVersion: '1337',
                eventSource: 's3:DummySource',
                eventName: 's3:DummyEvent',
                eventTime: '1789-07-14T00:00:00+02:00',
                userIdentity: { principalId: 'aws::principal::id' },
                requestParameters: { sourceIPAddress: '127.0.0.1' },
                responseElements: {
                    'x-amz-id-2': '456',
                    'x-amz-request-id': '123',
                },
                s3: {
                    bucket: {
                        name: 'dummy-bucket',
                        arn: 'arn:aws:s3:::dummy-bucket',
                        ownerIdentity: { principalId: 'aws::principal::id' },
                    },
                    configurationId: '42',
                    object: {
                        eTag: 'eTag',
                        key: `packages/${packageName}/v${packageVersion}/package.tgz`,
                        sequencer: 'Seq',
                        size: 1337,
                        versionId: 'VersionId',
                    },
                    s3SchemaVersion: '1',
                },
            }],
    };
    const mockContext = {};
    const mockTarballBytes = crypto_1.randomBytes(128);
    const mockOutputAssembly = crypto_1.randomBytes(128);
    AWSMock.mock('S3', 'getObject', (request, callback) => {
        try {
            expect(request.Bucket).toBe(payload.Records[0].s3.bucket.name);
            expect(request.Key).toBe(payload.Records[0].s3.object.key);
            expect(request.VersionId).toBe(payload.Records[0].s3.object.versionId);
        }
        catch (e) {
            callback(e);
        }
        callback(null, {
            Body: mockTarballBytes,
        });
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockSpawn = require('child_process').spawn;
    mockSpawn.mockImplementation((cmd, args, opts) => {
        expect(cmd).toBe('npm');
        expect(args).toContain('install');
        expect(args).toContain('--ignore-scripts'); // Ensures lifecycle hooks don't run
        expect(args).toContain('--no-bin-links'); // Ensures we don't attempt to add bin-links to $PATH
        expect(opts.cwd).toBeDefined();
        expect(opts.stdio).toEqual(['ignore', 'inherit', 'inherit']);
        const tarballPath = args[args.length - 1];
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        expect(require('fs').readFileSync(tarballPath)).toEqual(mockTarballBytes);
        return new MockChildProcess(cmd, Array.from(args), fs_1.promises.mkdir(path.join(opts.cwd, 'node_modules', packageName), { recursive: true }));
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockTransliterateAssembly = require('jsii-rosetta/lib/commands/transliterate').transliterateAssembly;
    mockTransliterateAssembly.mockImplementation(async ([dir, ...otherDirs], languages) => {
        expect(languages).toEqual([jsii_rosetta_1.TargetLanguage.PYTHON]);
        expect(dir).toMatch(new RegExp(path.join('', 'node_modules', packageName) + '$'));
        expect(otherDirs).toEqual([]);
        return fs_1.promises.writeFile(path.resolve(dir, '.jsii.python'), mockOutputAssembly);
    });
    const key = payload.Records[0].s3.object.key.replace(/\/package\.tgz$/, '/assembly-python.json');
    AWSMock.mock('S3', 'putObject', (request, callback) => {
        try {
            expect(request.Bucket).toBe(payload.Records[0].s3.bucket.name);
            expect(request.Key).toBe(key);
            expect(request.Body).toEqual(mockOutputAssembly);
        }
        catch (e) {
            return callback(e);
        }
        callback(null, { VersionId: 'New-VersionID' });
    });
    // WHEN
    const result = transliterator_lambda_1.handler(payload, mockContext);
    // THEN
    await expect(result).resolves.toEqual([{ bucket: payload.Records[0].s3.bucket.name, key, versionId: 'New-VersionID' }]);
    debugger;
    expect(mockSpawn).toHaveBeenCalled();
    expect(mockTransliterateAssembly).toHaveBeenCalled();
});
class MockChildProcess extends events_1.EventEmitter {
    constructor(spawnfile, spawnargs, promise) {
        super();
        this.spawnfile = spawnfile;
        this.spawnargs = spawnargs;
        this.stdin = null;
        this.stdout = null;
        this.stderr = null;
        this.stdio = [this.stdin, this.stdout, this.stderr, null, null];
        this.exitCode = 0;
        this.killed = false;
        this.signalCode = null;
        this.pid = -1; // Obviously fake
        promise.then(() => this.emit('close', this.exitCode, this.signalCode), (err) => this.emit('error', err));
    }
    get connected() {
        throw new Error('Not Implemented');
    }
    disconnect() {
        throw new Error('Not Implemented');
    }
    kill() {
        throw new Error('Not Implemented');
    }
    ref() {
        throw new Error('Not Implemented');
    }
    send() {
        throw new Error('Not Implemented');
    }
    unref() {
        throw new Error('Not Implemented');
    }
}
//# sourceMappingURL=data:application/json;base64,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