from __future__ import annotations

import sys
import builtins
from io import StringIO
from typing import Generator
from contextlib import contextmanager

import pytermgui as ptg

DEFAULT_WIDTH = 80
DEFAULT_HEIGHT = 24

EXEC_GLOBALS = {
    "__name__": "__main__",
    "__doc__": None,
    "__package__": None,
    "__annotations__": {},
    "__builtins__": builtins,
}


@contextmanager
def patched_stdout_recorder(
    width: int | None, height: int | None
) -> Generator[ptg.Recorder, None, None]:
    """Patches sys.stdout.write to write to a new terminal.

    Returns:
        The recorder that was written to.
    """

    if width is None:
        width = DEFAULT_WIDTH

    if height is None:
        height = DEFAULT_HEIGHT

    stdout_write = sys.stdout.write

    stream = StringIO()
    terminal = ptg.Terminal(stream=stream, size=(width, height))

    ptg.set_global_terminal(terminal)

    with terminal.record() as recorder:
        try:
            sys.stdout.write = terminal.write
            yield recorder

        finally:
            sys.stdout.write = stdout_write


def execute(
    module: str | None = None,
    file: Path | None = None,
    code: str | None = None,
    highlight: bool = None,
    *,
    exec_globals: dict[str, Any] = EXEC_GLOBALS
) -> None:
    """Executes the given code."""

    ptg.WindowManager.autorun = False

    if module is not None:
        # mod_name, *args = module.split()
        # sys.argv = [*args]
        # out = runpy.run_module(mod_name, init_globals={"sys": sys})
        # print(out)
        return

    if file is not None:
        with open(file, "r", encoding="utf-8") as source:
            code = source.read() + code

    if highlight:
        print(ptg.tim.parse(ptg.highlight_python(code)))
        return

    exec(code, exec_globals)

    if "manager" in exec_globals:
        exec_globals["manager"].compositor.draw()


def set_colors(foreground: str | None, background: str | None) -> None:
    """Sets the colors that will be used by the terminal."""

    if foreground is not None:
        ptg.Color.default_foreground = ptg.str_to_color(foreground)

    if background is not None:
        ptg.Color.default_background = ptg.str_to_color(background)
