# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['stake']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp>=3.6.2,<4.0.0',
 'inflection>=0.5.0,<0.6.0',
 'pydantic>=1.5.1,<2.0.0',
 'python-dotenv>=0.13.0,<0.14.0',
 'single-version>=1.2.2,<2.0.0']

setup_kwargs = {
    'name': 'stake',
    'version': '0.2.5',
    'description': 'Unofficial https://stake.com.au API wrapper.',
    'long_description': '[![pre-commit](https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white)](<https://github.com/pre-commit/pre-commit>)\n![Coverage](coverage.svg)\n\n# Stake\n\n**Stake** is an unofficial Python client for the [Stake](<https://www.stake.com.au>) trading platform.\n\nThis library wraps the current Stake RPC api and allows common trade operations, such as submitting buy/sell requests, checking your portfolio etc...\n\nPlease note that, at the current stage, the Stake client is asynchronous.\n\n## Installation\n\n~~~$\npip install stake\n~~~\n\n## Quickstart\n\nAfter you install the package, you will need to authenticate to Stake in order to get authorization to interact with your account.\nIn order to successfully issue requests to the Stake platform you will need to authenticate to it. Every requests to the Stake endpoints will need to contain a `Stake-Session-Token` in the request headers.\n\n## Using an existing Session-Token\n\nYou can retrieve one of these `Stake-Session-Token` by using the developer tools in your browser (Network tab) and inspecting some of the request headers sent to some of the `https://global-prd-api.hellostake.com/` host. (For example, click on the wishlist of dashboard links to see some session-token authenticated requests)\n\nThey are usually valid for 30 days (be sure to enable that checkbox on login) and seem to get refreshed before expiry so you should be good to use them directly.\n\nIf you already have an existing token you can pass it on to the `StakeClient` as such:\n\n~~~python\n\nfrom stake import StakeClient, SessionTokenLoginRequest, CredentialsLoginRequest\nimport asyncio\n\nlogin_request = SessionTokenLoginRequest()\nasync def print_user():\n    async with StakeClient(login_request) as stake_session:\n        print(stake_session.user.first_name)\n        print(stake_session.headers.stake_session_token)\n\nasyncio.run(print_user())\n~~~\n\n> **_NOTE:_**  The default value of the token is read from the `STAKE_TOKEN` environment variable. If you have that env-var set you should be able to just use:\n> `async with StakeClient() as stake_session: ...`\n\n## Login with your credentials\n\n> **_NOTE:_**  On November 2020 the Stake website introduced a *reCAPTCHA* protection on their api, therefore logging in through credentials is currently not working. Please log in using the *Session-Token* as described in previous section.\n\nIf you prefer to pass in your username/password credentials to login instead, it\'s easy to do:\n\n### If you do not have two-factor authentication enabled:\n\n~~~python\n\nfrom stake import StakeClient, SessionTokenLoginRequest, CredentialsLoginRequest\nimport asyncio\n\nlogin_request = CredentialsLoginRequest(\n    username="youruser@name.com", # os.getenv("STAKE_USER") by default\n    password="yoursecretpassword") # os.getenv("STAKE_PASS") by default\n\nasync def print_user():\n    async with StakeClient(login_request) as stake_session:\n        print(stake_session.user.first_name)\n        print(stake_session.headers.stake_session_token)\n\nasyncio.run(print_user())\n~~~\n\n### If you have two-factor authentication enabled:\n\nIn this case you need to have your phone around, get the current code from the authenticator app and add it to the `CredentialsLoginRequest` as such:\n\n~~~python\n    login_request = CredentialsLoginRequest(username="youruser@name.com",password="yoursecretpassword",\n        otp="Your-authenticator-app-code")\n~~~\n\nObviously, this can become a bit inconvenient, since you will need to provide the otp code every time you instantiate a new `StakeClient` instance. Therefore, you could probably authenticate once with your credentials, retrieve the session token from the headers(`stake_session.headers.stake_session_token`), and store it in the `STAKE_TOKEN` env-var for subsequent usages.\n\n## Examples\n\nWith `stake-python` you can do most of the operations that are available through the web app.\n\nHere are some examples:\n\n### Display the contents of your portfolio\n\n~~~python\nimport stake\nimport asyncio\n\n\nasync def show_portfolio():\n    # here the client will use the STAKE_TOKEN env var for authenticating\n    async with stake.StakeClient() as stake_session:\n        my_equities = await stake_session.equities.list()\n        for my_equity in my_equities.equity_positions:\n            print(my_equity.symbol, my_equity.yearly_return_value)\n        return my_equities\n\nasyncio.run(show_portfolio())\n~~~\n\nWhich will return something like:\n\n~~~\nAAPL 80.48\nADBE 251.35\nGOOG 559.89\nGRPN -13.77\nHTZ -10.52\nMSFT 97.14\nNFLX 263.55\nNIO 17.3\nNVDA 410.04\nOKTA 96.31\nSHOP 690.68\nSPOT 142.88\nSQ 101.75\nTQQQ 115.82\nTSLA 402.37\nVGT 130.08\nZM 331.1\n~~~\n\n### Buy/Sell shares\n\nYou can send buy/sell orders to the platform quite easily by just issuing trade requests.\nPlease check the `stake.trade` module for more details.\n\n~~~python\n\nimport asyncio\nimport stake\n\nasync def example_limit_buy():\n    symbol = "UNKN" # should be the equity symbol, for ex: AAPL, TSLA, GOOGL\n    async with stake.StakeClient() as stake_session:\n        return await stake_session.trades.buy(\n            stake.LimitBuyRequest(symbol=symbol, limitPrice=10, quantity=1000)\n        )\n\nasyncio.run(example_limit_buy())\n~~~\n\nTo perform multiple requests at once you can use an `asyncio.gather` operation to run all the buy trades in parallel.\n\n~~~python\n\nimport asyncio\nimport stake\n\nasync def example_stop_sell():\n    """THis example will add a stop sell request for one of your equities"""\n    async with stake.StakeClient() as stake_session:\n        my_equities = await stake_session.equities.list()\n        symbol = "TSLA" # mispelt on purpose so that no trade actually happens, should be TSLA.\n        tsla_equity = [equity for equity in my_equities.equity_positions if equity.symbol == symbol][0]\n        stop_price = round(tsla_equity.market_price - 0.025 * tsla_equity.market_price)\n        stop_sell_request = stake.StopSellRequest(symbol=tsla_equity.symbol,\n                                                  stopPrice=stop_price,\n                                                  comment="My stop sell.",\n                                                  quantity=tsla_equity.available_for_trading_qty)\n        return await stake_session.trades.sell(request=stop_sell_request)\n\nasyncio.run(example_stop_sell())\n~~~\n\n## Contributors\n\n### Contributors on GitHub\n\n* [Contributors](<https://github.com/stabacco/stake-python/graphs/contributors>)\n\n## License\n\n* see [LICENSE](<https://github.com/stabacco/stake-python/blob/master/LICENSE.md>) file\n\n## Contact\n\n* Created by [Stefano Tabacco](<https://github.com/stabacco>)\n',
    'author': 'Stefano Tabacco',
    'author_email': 'tabacco.stefano@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/stabacco/stake-python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
