from __future__ import annotations

import random
import time

from pathlib import Path
from typing import Iterable, Union, Type

import click
import git
import github
import yaml.parser

from github.Repository import Repository

from commodore.config import Config
from commodore.helpers import yaml_load

from commodore.component import Component
from commodore.package import Package

from commodore.dependency_templater import Templater


def sync_dependencies(
    config: Config,
    dependency_list: Path,
    dry_run: bool,
    pr_branch: str,
    pr_label: Iterable[str],
    deptype: Type[Union[Component, Package]],
    templater: Type[Templater],
) -> None:
    if not config.github_token:
        raise click.ClickException("Can't continue, missing GitHub API token.")

    deptype_str = deptype.__name__.lower()

    try:
        deps = yaml_load(dependency_list)
        if not isinstance(deps, list):
            raise ValueError(f"unexpected type: {type_name(deps)}")
    except ValueError as e:
        raise click.ClickException(
            f"Expected a list in '{dependency_list}', but got {e}"
        )
    except (yaml.parser.ParserError, yaml.scanner.ScannerError):
        raise click.ClickException(f"Failed to parse YAML in '{dependency_list}'")

    gh = github.Github(config.github_token)
    dep_count = len(deps)
    for i, dn in enumerate(deps, start=1):
        click.secho(f"Synchronizing {dn}", bold=True)
        _, dreponame = dn.split("/")
        dname = dreponame.replace(f"{deptype_str}-", "", 1)

        # Clone dependency
        try:
            gr = gh.get_repo(dn)
        except github.UnknownObjectException:
            click.secho(f" > Repository {dn} doesn't exist, skipping...", fg="yellow")
            continue
        d = deptype.clone(config, gr.clone_url, dname, version=gr.default_branch)

        if not (d.target_dir / ".cruft.json").is_file():
            click.echo(f" > Skipping repo {dn} which doesn't have `.cruft.json`")
            continue

        # Update the dependency
        t = templater.from_existing(config, d.target_dir)
        changed = t.update(print_completion_message=False, commit=not dry_run)

        # Create or update PR if there were updates
        create_or_update_pr(d, dn, gr, changed, pr_branch, pr_label, dry_run)
        if changed and not dry_run and i < dep_count:
            # except when processing the last dependency in the list, sleep for 1-2
            # seconds to avoid hitting secondary rate-limits for PR creation. No
            # need to sleep if we're not creating a PR.
            # Without the #nosec annotations bandit warns (correctly) that
            # `random.random()` generates weak random numbers, but since the quality
            # of the randomness doesn't matter here, we don't need to use a more
            # expensive RNG.
            backoff = 1.0 + random.random()  # nosec
            time.sleep(backoff)


def create_or_update_pr(
    d: Union[Component, Package],
    dn: str,
    gr: Repository,
    changed: bool,
    pr_branch: str,
    pr_label,
    dry_run: bool,
):
    if dry_run and changed:
        click.secho(f"Would create or update PR for {dn}", bold=True)
    elif changed:
        ensure_branch(d, pr_branch)
        msg = ensure_pr(d, dn, gr, pr_branch, pr_label)
        click.secho(msg, bold=True)
    else:
        dep_type = type_name(d)
        click.secho(f"{dep_type.capitalize()} {dn} already up-to-date", bold=True)


def message_body(c: git.objects.commit.Commit) -> str:
    if isinstance(c.message, bytes):
        msg = str(c.message, encoding="utf-8")
    else:
        msg = c.message
    paragraphs = msg.split("\n\n")
    return "\n\n".join(paragraphs[1:])


def ensure_branch(d: Union[Component, Package], branch_name: str):
    """Create or reset `template-sync` branch pointing to our new template update
    commit."""
    deptype = type_name(d)

    if not d.repo:
        raise ValueError(f"{deptype} repo not initialized")
    r = d.repo.repo
    has_sync_branch = any(h.name == branch_name for h in r.heads)

    if not has_sync_branch:
        r.create_head(branch_name)
    else:
        new_update = r.head.commit
        template_sync = [h for h in r.heads if h.name == branch_name][0]
        template_sync.set_reference(new_update)


def ensure_pr(
    d: Union[Component, Package],
    dn: str,
    gr: Repository,
    branch_name: str,
    pr_labels: Iterable[str],
) -> str:
    """Create or update template sync PR."""
    deptype = type_name(d)

    if not d.repo:
        raise ValueError(f"{deptype} repo not initialized")

    prs = gr.get_pulls(state="open")
    has_sync_pr = any(pr.head.ref == branch_name for pr in prs)
    cu = "update" if has_sync_pr else "create"

    r = d.repo.repo
    r.remote().push(branch_name, force=True)
    pr_body = message_body(r.head.commit)

    try:
        if not has_sync_pr:
            sync_pr = gr.create_pull(
                f"Update from {deptype} template",
                pr_body,
                gr.default_branch,
                branch_name,
            )
        else:
            sync_pr = [pr for pr in prs if pr.head.ref == branch_name][0]
            sync_pr.edit(body=pr_body)
        sync_pr.add_to_labels(*list(pr_labels))
    except github.UnknownObjectException:
        return (
            f"Unable to {cu} PR for {dn}. "
            + "Please make sure your GitHub token has permission 'public_repo'"
        )

    return f"PR for {deptype} {dn} successfully {cu}d"


def type_name(o: object) -> str:
    return type(o).__name__.lower()
