"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cf = require("@aws-cdk/aws-cloudfront");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const iam = require("@aws-cdk/aws-iam");
const firehose = require("@aws-cdk/aws-kinesisfirehose");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const extensions = require("../../extensions");
const deliveryStreamName = 'test';
/**
 * CDK App and Stack
 */
const app = new cdk.App();
const stack = new cdk.Stack(app, 'global-data-ingestion');
/**
 * Kinesis Firehose components
 */
const firehoseRole = new iam.Role(stack, 'FirehoseRole', {
    assumedBy: new iam.ServicePrincipal('firehose.amazonaws.com'),
});
const firehoseDataBucket = new s3.Bucket(stack, 'FirehoseDataBucket', {
    autoDeleteObjects: true,
    removalPolicy: cdk.RemovalPolicy.DESTROY,
});
firehoseDataBucket.grantReadWrite(firehoseRole);
const firehoseStream = new firehose.CfnDeliveryStream(stack, 'FirehoseStream', {
    deliveryStreamName,
    deliveryStreamType: 'DirectPut',
    extendedS3DestinationConfiguration: {
        bucketArn: firehoseDataBucket.bucketArn,
        bufferingHints: {
            intervalInSeconds: 60,
            sizeInMBs: 5,
        },
        roleArn: firehoseRole.roleArn,
        errorOutputPrefix: 'error',
    },
});
/**
 * Cloudfront and Lambda@Edge
 */
const ext = new extensions.GlobalDataIngestion(stack, 'GlobalDataIngestion', {
    firehoseStreamName: firehoseStream.deliveryStreamName,
});
const dist = new cf.Distribution(stack, 'dist', {
    defaultBehavior: {
        allowedMethods: cf.AllowedMethods.ALLOW_ALL,
        origin: new origins.HttpOrigin('aws.amazon.com'),
        edgeLambdas: [ext],
    },
});
/**
 * Outputs
 */
new cdk.CfnOutput(stack, 'distributionDomainName', {
    value: dist.distributionDomainName,
});
new cdk.CfnOutput(stack, 'firehoseStreamName', {
    value: firehoseStream.deliveryStreamName,
});
new cdk.CfnOutput(stack, 'firehoseDataBucket', {
    value: firehoseDataBucket.bucketName,
});
//# sourceMappingURL=data:application/json;base64,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