"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const cf = require("@aws-cdk/aws-cloudfront");
const s3 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const cdk = require("@aws-cdk/core");
const extensions = require("../../extensions");
const app = new cdk.App();
const stack = new cdk.Stack(app, 'demo-redirect-error-page');
// create the cloudfront distribution with extension(s)
const customErrorPage = new extensions.CustomErrorPage(stack, 'customErrorPage');
// create s3 bucket
const bucket = new s3.Bucket(customErrorPage, 'demoBucket', {
    autoDeleteObjects: true,
    removalPolicy: cdk.RemovalPolicy.DESTROY,
    websiteIndexDocument: 'index.html',
    websiteErrorDocument: 'error.html',
});
// create pages
fs.writeFileSync(path.join(__dirname, 'index.html'), '<h1>Hello CDK!</h1>');
fs.writeFileSync(path.join(__dirname, 'error.html'), '<h1>This is an ERROR.</h1>');
fs.writeFileSync(path.join(__dirname, '404.html'), '<h1>This is a custom 404 error page.</h1>');
// put pages to s3 bucket
new aws_s3_deployment_1.BucketDeployment(customErrorPage, 'Deployment', {
    sources: [aws_s3_deployment_1.Source.asset(path.join(__dirname, './'))],
    destinationBucket: bucket,
    retainOnDelete: false,
});
// cloudFront OriginAccessIdentity for bucket
const originAccessIdentity = new cf.OriginAccessIdentity(customErrorPage, 'OriginAccessIdentity', {
    comment: `CloudFront OriginAccessIdentity for ${bucket.bucketName}`,
});
// cloudfront distribution
const distribution = new cf.CloudFrontWebDistribution(stack, 'distribution', {
    originConfigs: [
        {
            s3OriginSource: {
                originAccessIdentity,
                s3BucketSource: bucket,
            },
            behaviors: [{
                    isDefaultBehavior: true,
                    defaultTtl: cdk.Duration.seconds(10),
                    lambdaFunctionAssociations: [customErrorPage],
                }],
        },
    ],
});
new cdk.CfnOutput(stack, 'distributionDomainName', {
    value: distribution.distributionDomainName,
});
//# sourceMappingURL=data:application/json;base64,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