"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const cf = require("@aws-cdk/aws-cloudfront");
const s3 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const cdk = require("@aws-cdk/core");
const dotenv = require("dotenv");
const extensions = require("../../extensions");
const resultDotEnv = dotenv.config();
if (resultDotEnv.error) {
    throw resultDotEnv.error;
}
const app = new cdk.App();
const stack = new cdk.Stack(app, 'cf-authentication-by-oauth2-demo');
// create the cloudfront distribution with extension(s)
const OAuth2AuthorizationCodeGrant = new extensions.OAuth2AuthorizationCodeGrant(stack, 'OAuth2AuthorizationCodeGrant', {
    clientId: process.env.CLIENT_ID,
    clientSecret: process.env.CLIENT_SECRET,
    clientDomain: process.env.CLIENT_DOMAIN,
    clientPublicKey: new Buffer(process.env.CLIENT_PUBLIC_KEY).toString('base64'),
    callbackPath: process.env.CALLBACK_PATH,
    jwtArgorithm: process.env.JWT_ARGORITHM,
    authorizeUrl: process.env.AUTHORIZE_URL,
    authorizeParams: new Buffer(process.env.AUTHORIZE_PARAMS).toString('base64'),
    debugEnable: process.env.DEBUG_ENABLE,
});
// create Demo S3 Bucket.
const bucket = new s3.Bucket(stack, 'demoBucket', {
    autoDeleteObjects: true,
    removalPolicy: cdk.RemovalPolicy.DESTROY,
    websiteIndexDocument: 'index.html',
    websiteErrorDocument: 'index.html',
});
// create index.html in the demo folder
fs.writeFileSync(path.join(__dirname, 'index.html'), '<h1>Hello CloudFront Extension (OAuth2 Authentication) with CDK!!!</h1><p>You have logged in. Enjoy your private content.</p>');
// Put demo Object to Bucket.
new aws_s3_deployment_1.BucketDeployment(stack, 'BucketDeployment', {
    sources: [aws_s3_deployment_1.Source.asset(path.join(__dirname, './'))],
    destinationBucket: bucket,
    retainOnDelete: false,
});
// CloudFront OriginAccessIdentity for Bucket
const originAccessIdentity = new cf.OriginAccessIdentity(stack, 'OriginAccessIdentity', {
    comment: `CloudFront OriginAccessIdentity for ${bucket.bucketName}`,
});
// CloudfrontWebDistribution
const cloudfrontWebDistribution = new cf.CloudFrontWebDistribution(stack, 'CloudFrontWebDistribution', {
    originConfigs: [
        {
            s3OriginSource: {
                originAccessIdentity,
                s3BucketSource: bucket,
            },
            behaviors: [{
                    isDefaultBehavior: true,
                    lambdaFunctionAssociations: [OAuth2AuthorizationCodeGrant],
                }],
        },
    ],
    priceClass: cf.PriceClass.PRICE_CLASS_ALL,
});
new cdk.CfnOutput(stack, 'distributionDomainName', {
    value: cloudfrontWebDistribution.distributionDomainName,
});
//# sourceMappingURL=data:application/json;base64,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