"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const cf = require("@aws-cdk/aws-cloudfront");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const iam = require("@aws-cdk/aws-iam");
const firehose = require("@aws-cdk/aws-kinesisfirehose");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const extensions = require("../extensions");
test('minimal usage', () => {
    // GIVEN
    const deliveryStreamName = 'test';
    const app = new cdk.App();
    const stack = new cdk.Stack(app, 'global-data-ingestion');
    // WHEN
    /**
     * Kinesis Firehose components
     */
    const firehoseRole = new iam.Role(stack, 'FirehoseRole', {
        assumedBy: new iam.ServicePrincipal('firehose.amazonaws.com'),
    });
    const firehoseDataBucket = new s3.Bucket(stack, 'FirehoseDataBucket', {
        autoDeleteObjects: true,
        removalPolicy: cdk.RemovalPolicy.DESTROY,
    });
    firehoseDataBucket.grantReadWrite(firehoseRole);
    const firehoseStream = new firehose.CfnDeliveryStream(stack, 'FirehoseStream', {
        deliveryStreamName,
        deliveryStreamType: 'DirectPut',
        extendedS3DestinationConfiguration: {
            bucketArn: firehoseDataBucket.bucketArn,
            bufferingHints: {
                intervalInSeconds: 60,
                sizeInMBs: 5,
            },
            roleArn: firehoseRole.roleArn,
            errorOutputPrefix: 'error',
        },
    });
    /**
     * Cloudfront and Lambda@Edge
     */
    const ext = new extensions.GlobalDataIngestion(stack, 'GlobalDataIngestion', {
        firehoseStreamName: firehoseStream.deliveryStreamName,
    });
    const dist = new cf.Distribution(stack, 'dist', {
        defaultBehavior: {
            allowedMethods: cf.AllowedMethods.ALLOW_ALL,
            origin: new origins.HttpOrigin('aws.amazon.com'),
            edgeLambdas: [ext],
        },
    });
    /**
     * Outputs
     */
    new cdk.CfnOutput(stack, 'distributionDomainName', {
        value: dist.distributionDomainName,
    });
    new cdk.CfnOutput(stack, 'firehoseStreamName', {
        value: firehoseStream.deliveryStreamName,
    });
    new cdk.CfnOutput(stack, 'firehoseDataBucket', {
        value: firehoseDataBucket.bucketName,
    });
    // THEN
    expect(assert_1.SynthUtils.synthesize(stack).template).toMatchSnapshot();
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                    'OPTIONS',
                    'PUT',
                    'PATCH',
                    'POST',
                    'DELETE',
                ],
                LambdaFunctionAssociations: [
                    {
                        EventType: 'viewer-request',
                        IncludeBody: true,
                        LambdaFunctionARN: {
                            Ref: 'GlobalDataIngestionFuncCurrentVersion202047486fb2c8727f1e9d4567ff9bf06110a3f2',
                        },
                    },
                ],
                ViewerProtocolPolicy: 'allow-all',
            },
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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