"""

Copyright (C) 2017-2020 Vanessa Sochat.

This Source Code Form is subject to the terms of the
Mozilla Public License, v. 2.0. If a copy of the MPL was not distributed
with this file, You can obtain one at http://mozilla.org/MPL/2.0/.

"""

from helpme.main import HelperBase
from helpme.action import record_asciinema, upload_asciinema
from helpme.logger import bot
import uservoice
import os
import sys


class Helper(HelperBase):
    def __init__(self, **kwargs):

        self.name = "uservoice"
        super(Helper, self).__init__(**kwargs)

    def load_secrets(self):

        required_vars = ["subdomain", "api_key", "api_secret", "email"]
        for required in required_vars:
            envar = "HELPME_USERVOICE_%s" % required.upper()

            # The settings can be provided in either config, depends on install

            user_setting = self._get_and_update_setting(envar, user=True)
            setting = self._get_and_update_setting(envar, user=False)
            setting = setting or user_setting

            if not setting:
                bot.error("export %s environment or add to helpme.cfg" % envar)
                sys.exit(1)

            setattr(self, required, setting)

    def _submit(self):
        """submit a uservoice ticket. When we get here we should have:
           
           {'user_prompt_issue': 'I want to do the thing.', 
            'record_asciinema': '/tmp/helpme.93o__nt5.json',
            'record_environment': ((1,1),(2,2)...(N,N))}

           Required Client Variables
           self.api_key
           self.api_secret
           self.subdomain
           self.email

        """

        # Step 0: Authenticate with uservoice API
        self.authenticate()

        title = "HelpMe UserVoice Ticket: %s" % (self.run_id)
        body = self.data["user_prompt_issue"]

        # Step 1: Environment

        envars = self.data.get("record_environment")
        if envars not in [None, "", []]:
            body += "\n\nEnvironment:\n"
            for envar in envars:
                body += " - %s: %s\n" % (envar[0], envar[1])

        # Step 2: Asciinema

        asciinema = self.data.get("record_asciinema")
        if asciinema not in [None, ""]:
            url = upload_asciinema(asciinema)

            # If the upload is successful, add a link to it.

            if url is not None:
                body += "\n\nAsciinema Recording: %s" % url

        # Add other metadata about client

        body += "\ngenerated by HelpMe: https://vsoch.github.io/helpme/"

        # Submit the ticket!

        self.post_ticket(title, body)

    # UserVoice API Helpers

    def authenticate(self):
        """authenticate with uservoice by creating a client."""

        if not hasattr(self, "client"):
            self.client = uservoice.Client(
                self.subdomain, self.api_key, self.api_secret
            )

    def post_ticket(self, title, body):
        """post_ticket will post a ticket to the uservoice helpdesk

           Parameters
           ==========
           title: the title (subject) of the issue
           body: the message to send

        """

        # Populate the ticket
        ticket = {"subject": title, "message": body}

        response = self.client.post(
            "/api/v1/tickets.json", {"email": self.email, "ticket": ticket}
        )["ticket"]
        bot.info(response["url"])
