"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.directorySnapshot = exports.synthSnapshot = void 0;
const os = require("os");
const path = require("path");
const fs = require("fs-extra");
const glob_1 = require("glob");
/**
 * Creates a snapshot of the files generated by a project. Ignores any non-text
 * files so that the snapshots are human readable.
 */
function synthSnapshot(project) {
    // defensive: verify that "outdir" is actually in a temporary directory
    if (!path.resolve(project.outdir).startsWith(os.tmpdir()) && !project.outdir.includes('project-temp-dir')) {
        throw new Error('Trying to capture a snapshot of a project outside of tmpdir, which implies this test might corrupt an existing project');
    }
    const synthed = Symbol.for('synthed');
    if (synthed in project) {
        throw new Error('duplicate synth()');
    }
    project[synthed] = true;
    const ENV_PROJEN_DISABLE_POST = process.env.PROJEN_DISABLE_POST;
    try {
        process.env.PROJEN_DISABLE_POST = 'true';
        project.synth();
        const ignoreExts = ['png', 'ico'];
        return directorySnapshot(project.outdir, { excludeGlobs: ignoreExts.map(ext => `**/*.${ext}`) });
    }
    finally {
        fs.removeSync(project.outdir);
        // values assigned to process.env.XYZ are automatically converted to strings
        if (ENV_PROJEN_DISABLE_POST === undefined) {
            delete process.env.PROJEN_DISABLE_POST;
        }
        else {
            process.env.PROJEN_DISABLE_POST = ENV_PROJEN_DISABLE_POST;
        }
    }
}
exports.synthSnapshot = synthSnapshot;
function directorySnapshot(root, options = {}) {
    var _a;
    const output = {};
    const files = glob_1.glob.sync('**', {
        ignore: ['.git/**', ...((_a = options.excludeGlobs) !== null && _a !== void 0 ? _a : [])],
        cwd: root,
        nodir: true,
        dot: true,
    }); // returns relative file paths with POSIX separators
    for (const file of files) {
        const filePath = path.join(root, file);
        let content;
        if (path.extname(filePath) === '.json') {
            content = fs.readJsonSync(filePath);
        }
        else {
            content = fs.readFileSync(filePath, 'utf-8');
        }
        output[file] = content;
    }
    return output;
}
exports.directorySnapshot = directorySnapshot;
//# sourceMappingURL=data:application/json;base64,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