"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.tryReadFile = exports.kebabCaseKeys = exports.formatAsPythonModule = exports.sorted = exports.dedupArray = exports.deepMerge = exports.isObject = exports.isTruthy = exports.decamelizeKeysRecursively = exports.writeFile = exports.getFilePermissions = exports.execOrUndefined = exports.execCapture = exports.exec = void 0;
const child_process = require("child_process");
const path = require("path");
const Case = require("case");
const fs = require("fs-extra");
const logging = require("./logging");
const MAX_BUFFER = 10 * 1024 * 1024;
/**
 * Executes a command with STDOUT > STDERR.
 */
function exec(command, options) {
    logging.debug(command);
    child_process.execSync(command, {
        stdio: ['inherit', process.stderr, 'pipe'],
        maxBuffer: MAX_BUFFER,
        cwd: options.cwd,
    });
}
exports.exec = exec;
/**
 * Executes command and returns STDOUT. If the command fails (non-zero), throws an error.
 */
function execCapture(command, options) {
    logging.debug(command);
    return child_process.execSync(command, {
        stdio: ['inherit', 'pipe', 'pipe'],
        maxBuffer: MAX_BUFFER,
        cwd: options.cwd,
    });
}
exports.execCapture = execCapture;
/**
 * Executes `command` and returns its value or undefined if the command failed.
 */
function execOrUndefined(command, options) {
    try {
        const value = child_process.execSync(command, {
            stdio: ['inherit', 'pipe', 'pipe'],
            maxBuffer: MAX_BUFFER,
            cwd: options.cwd,
        }).toString('utf-8').trim();
        if (!value) {
            return undefined;
        } // an empty string is the same as undefined
        return value;
    }
    catch {
        return undefined;
    }
}
exports.execOrUndefined = execOrUndefined;
function getFilePermissions(options) {
    var _a, _b;
    const readonly = (_a = options.readonly) !== null && _a !== void 0 ? _a : false;
    const executable = (_b = options.executable) !== null && _b !== void 0 ? _b : false;
    if (readonly && executable) {
        return '500';
    }
    else if (readonly) {
        return '400';
    }
    else if (executable) {
        return '755';
    }
    else {
        return '644';
    }
}
exports.getFilePermissions = getFilePermissions;
function writeFile(filePath, data, options = {}) {
    if (fs.existsSync(filePath)) {
        fs.chmodSync(filePath, '600');
    }
    fs.mkdirpSync(path.dirname(filePath));
    fs.writeFileSync(filePath, data);
    fs.chmodSync(filePath, getFilePermissions(options));
}
exports.writeFile = writeFile;
function decamelizeKeysRecursively(input, opt) {
    var _a, _b, _c, _d;
    const shouldAlwaysDecamelize = () => true;
    const shouldDecamelize = (_a = opt === null || opt === void 0 ? void 0 : opt.shouldDecamelize) !== null && _a !== void 0 ? _a : shouldAlwaysDecamelize;
    const separator = (_b = opt === null || opt === void 0 ? void 0 : opt.separator) !== null && _b !== void 0 ? _b : '_';
    const path_ = (_c = opt === null || opt === void 0 ? void 0 : opt.path) !== null && _c !== void 0 ? _c : [];
    const maxDepth = (_d = opt === null || opt === void 0 ? void 0 : opt.maxDepth) !== null && _d !== void 0 ? _d : 10;
    if (path_.length > maxDepth) {
        throw new Error('Decamelled too deeply - check that the input has no circular references');
    }
    if (Array.isArray(input)) {
        return input.map((k, i) => decamelizeKeysRecursively(k, {
            ...opt,
            path: [...path_, i.toString()],
        }));
    }
    if (typeof input === 'object' && input !== null) {
        const mappedObject = {};
        for (const [key, value] of Object.entries(input)) {
            const transformedKey = shouldDecamelize([...path_, key], value)
                ? decamelize(key, separator)
                : key;
            mappedObject[transformedKey] = decamelizeKeysRecursively(value, {
                ...opt,
                path: [...path_, key],
            });
        }
        return mappedObject;
    }
    return input;
}
exports.decamelizeKeysRecursively = decamelizeKeysRecursively;
/**
 * Returns false if value is unset or a falsey value, and true otherwise.
 * @param value an environment variable
 */
function isTruthy(value) {
    return !(value === undefined || ['null', 'undefined', '0', 'false', ''].includes(value.toLocaleLowerCase()));
}
exports.isTruthy = isTruthy;
/**
 * Return whether the given value is an object
 *
 * Even though arrays and instances of classes technically are objects, we
 * usually want to treat them differently, so we return false in those cases.
 */
function isObject(x) {
    return x !== null && typeof x === 'object' && !Array.isArray(x)
        && x.constructor.name === 'Object';
}
exports.isObject = isObject;
/**
 * Recursively merge objects together
 *
 * The leftmost object is mutated and returned. Arrays are not merged
 * but overwritten just like scalars.
 *
 * If an object is merged into a non-object, the non-object is lost.
 *
 * `undefined`s will cause a value to be deleted if destructive is enabled.
 */
function deepMerge(objects, destructive = false) {
    function mergeOne(target, source) {
        for (const key of Object.keys(source)) {
            const value = source[key];
            if (isObject(value)) {
                // if the value at the target is not an object, override it with an
                // object so we can continue the recursion
                if (typeof (target[key]) !== 'object') {
                    target[key] = value;
                }
                mergeOne(target[key], value);
                // if the result of the merge is an empty object, it's because the
                // eventual value we assigned is `undefined`, and there are no
                // sibling concrete values alongside, so we can delete this tree.
                const output = target[key];
                if (typeof (output) === 'object' && Object.keys(output).length === 0 && destructive) {
                    delete target[key];
                }
            }
            else if (value === undefined && destructive) {
                delete target[key];
            }
            else if (typeof value !== 'undefined') {
                target[key] = value;
            }
        }
    }
    const others = objects.filter(x => x != null);
    if (others.length === 0) {
        return {};
    }
    const into = others.splice(0, 1)[0];
    others.forEach(other => mergeOne(into, other));
    return into;
}
exports.deepMerge = deepMerge;
/*
 * Deduplicate values in a list, returning a new array.
 * @param array list of values
 */
function dedupArray(array) {
    return array.filter((val, idx) => array.indexOf(val) === idx);
}
exports.dedupArray = dedupArray;
/**
 * Returns a sorted version of `x` or `undefined` if it is an empty array or object.
 */
function sorted(x) {
    if (x == null) {
        return undefined;
    }
    if (Array.isArray(x)) {
        if (x.length === 0) {
            return undefined;
        }
        return x.sort();
    }
    else if (typeof x === 'object') {
        if (Object.keys(x).length === 0) {
            return undefined;
        }
        const result = {};
        for (const [key, value] of Object.entries(x).sort(([l], [r]) => l.localeCompare(r))) {
            result[key] = value;
        }
        return result;
    }
    else {
        return x;
    }
}
exports.sorted = sorted;
function formatAsPythonModule(name) {
    return name.replace(/-/g, '_').replace(/\./g, '_');
}
exports.formatAsPythonModule = formatAsPythonModule;
function kebabCaseKeys(obj, recursive = true) {
    if (typeof obj !== 'object' || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        if (recursive) {
            obj = obj.map((v) => kebabCaseKeys(v, recursive));
        }
        return obj;
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (recursive) {
            v = kebabCaseKeys(v, recursive);
        }
        result[decamelize(k).replace(/_/mg, '-')] = v;
    }
    return result;
}
exports.kebabCaseKeys = kebabCaseKeys;
async function tryReadFile(file) {
    if (!(await fs.pathExists(file))) {
        return '';
    }
    return fs.readFile(file, 'utf8');
}
exports.tryReadFile = tryReadFile;
function decamelize(s, sep = '_') {
    if (Case.of(s) === 'camel') {
        return Case.lower(s, sep);
    }
    else {
        return s;
    }
}
;
//# sourceMappingURL=data:application/json;base64,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