"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TaskRuntime = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const child_process_1 = require("child_process");
const fs_1 = require("fs");
const os_1 = require("os");
const path_1 = require("path");
const util_1 = require("util");
const chalk = require("chalk");
const logging = require("./logging");
const tasks_1 = require("./tasks");
const ENV_TRIM_LEN = 20;
/**
 * (experimental) The runtime component of the tasks engine.
 *
 * @experimental
 */
class TaskRuntime {
    /**
     * @experimental
     */
    constructor(workdir) {
        this.workdir = path_1.resolve(workdir);
        const manifestPath = path_1.join(this.workdir, tasks_1.Tasks.MANIFEST_FILE);
        this.manifest = fs_1.existsSync(manifestPath)
            ? JSON.parse(fs_1.readFileSync(manifestPath, 'utf-8'))
            : { tasks: {} };
    }
    /**
     * (experimental) The tasks in this project.
     *
     * @experimental
     */
    get tasks() {
        var _b;
        return Object.values((_b = this.manifest.tasks) !== null && _b !== void 0 ? _b : {});
    }
    /**
     * (experimental) Find a task by name, or `undefined` if not found.
     *
     * @experimental
     */
    tryFindTask(name) {
        if (!this.manifest.tasks) {
            return undefined;
        }
        return this.manifest.tasks[name];
    }
    /**
     * (experimental) Runs the task.
     *
     * @param name The task name.
     * @experimental
     */
    runTask(name, parents = []) {
        const task = this.tryFindTask(name);
        if (!task) {
            throw new Error(`cannot find command ${task}`);
        }
        new RunTask(this, task, parents);
    }
}
exports.TaskRuntime = TaskRuntime;
_a = JSII_RTTI_SYMBOL_1;
TaskRuntime[_a] = { fqn: "projen.TaskRuntime", version: "0.40.2" };
class RunTask {
    constructor(runtime, task, parents = []) {
        var _b, _c, _d;
        this.runtime = runtime;
        this.task = task;
        this.env = {};
        this.workdir = (_b = task.cwd) !== null && _b !== void 0 ? _b : this.runtime.workdir;
        this.parents = parents;
        if (!task.steps || task.steps.length === 0) {
            this.logDebug(chalk.gray('No actions have been specified for this task.'));
            return;
        }
        this.env = this.resolveEnvironment(parents);
        const envlogs = [];
        for (const [k, v] of Object.entries(this.env)) {
            const vv = v !== null && v !== void 0 ? v : '';
            const trimmed = vv.length > ENV_TRIM_LEN ? vv.substr(0, ENV_TRIM_LEN) + '...' : vv;
            envlogs.push(`${k}=${trimmed}`);
        }
        if (envlogs.length) {
            this.logDebug(chalk.gray(`${chalk.underline('env')}: ${envlogs.join(' ')}`));
        }
        // evaluate condition
        if (!this.evalCondition(task)) {
            this.log('condition exited with non-zero - skipping');
            return;
        }
        // verify we required environment variables are defined
        const merged = { ...process.env, ...this.env };
        const missing = new Array();
        for (const name of (_c = task.requiredEnv) !== null && _c !== void 0 ? _c : []) {
            if ((!(name in merged))) {
                missing.push(name);
            }
        }
        if (missing.length > 0) {
            throw new Error(`missing required environment variables: ${missing.join(',')}`);
        }
        for (const step of task.steps) {
            if (step.say) {
                logging.info(this.fmtLog(step.say));
            }
            if (step.spawn) {
                this.runtime.runTask(step.spawn, [...this.parents, this.task.name]);
            }
            const execs = step.exec ? [step.exec] : [];
            if (step.builtin) {
                execs.push(this.renderBuiltin(step.builtin));
            }
            for (const exec of execs) {
                let command = '';
                let hasError = false;
                const cmd = exec.split(' ')[0];
                if (os_1.platform() == 'win32' && ['mkdir', 'mv', 'rm'].includes(cmd)) {
                    command = `shx ${exec}`;
                }
                else {
                    command = exec;
                }
                const cwd = step.cwd;
                try {
                    const result = this.shell({
                        command,
                        cwd,
                    });
                    hasError = result.status !== 0;
                }
                catch (e) {
                    // This is the error 'shx' will throw
                    if ((_d = e === null || e === void 0 ? void 0 : e.message) === null || _d === void 0 ? void 0 : _d.startsWith('non-zero exit code:')) {
                        hasError = true;
                    }
                    throw e;
                }
                if (hasError) {
                    throw new Error(`Task "${this.fullname}" failed when executing "${command}" (cwd: ${path_1.resolve(cwd !== null && cwd !== void 0 ? cwd : this.workdir)})`);
                }
            }
        }
    }
    /**
     * Determines if a task should be executed based on "condition".
     *
     * @returns true if the task should be executed or false if the condition
     * evaluates to false (exits with non-zero), indicating that the task should
     * be skipped.
     */
    evalCondition(task) {
        // no condition, carry on
        if (!task.condition) {
            return true;
        }
        this.log(chalk.gray(`${chalk.underline('condition')}: ${task.condition}`));
        const result = this.shell({
            command: task.condition,
            logprefix: 'condition: ',
            quiet: true,
        });
        if (result.status === 0) {
            return true;
        }
        else {
            return false;
        }
    }
    /**
     * Renders the runtime environment for a task. Namely, it supports this syntax
     * `$(xx)` for allowing environment to be evaluated by executing a shell
     * command and obtaining its result.
     */
    resolveEnvironment(parents) {
        var _b, _c, _d, _e, _f, _g;
        let env = (_b = this.runtime.manifest.env) !== null && _b !== void 0 ? _b : {};
        // add env from all parent tasks one by one
        for (const parent of parents) {
            env = {
                ...env,
                ...(_d = (_c = this.runtime.tryFindTask(parent)) === null || _c === void 0 ? void 0 : _c.env) !== null && _d !== void 0 ? _d : {},
            };
        }
        // apply the task's environment last
        env = {
            ...env,
            ...(_e = this.task.env) !== null && _e !== void 0 ? _e : {},
        };
        const output = {};
        for (const [key, value] of Object.entries(env !== null && env !== void 0 ? env : {})) {
            if (value.startsWith('$(') && value.endsWith(')')) {
                const query = value.substring(2, value.length - 1);
                const result = this.shellEval({ command: query });
                if (result.status !== 0) {
                    const error = result.error
                        ? result.error.stack
                        : (_g = (_f = result.stderr) === null || _f === void 0 ? void 0 : _f.toString()) !== null && _g !== void 0 ? _g : 'unknown error';
                    throw new Error(`unable to evaluate environment variable ${key}=${value}: ${error}`);
                }
                output[key] = result.stdout.toString('utf-8').trim();
            }
            else {
                output[key] = value;
            }
        }
        return output;
    }
    /**
     * Returns the "full name" of the task which includes all it's parent task names concatenated by chevrons.
     */
    get fullname() {
        return [...this.parents, this.task.name].join(' » ');
    }
    log(...args) {
        logging.verbose(this.fmtLog(...args));
    }
    logDebug(...args) {
        logging.debug(this.fmtLog(...args));
    }
    fmtLog(...args) {
        return util_1.format(`${chalk.underline(this.fullname)} |`, ...args);
    }
    shell(options) {
        var _b, _c;
        const quiet = (_b = options.quiet) !== null && _b !== void 0 ? _b : false;
        if (!quiet) {
            const log = new Array();
            if (options.logprefix) {
                log.push(options.logprefix);
            }
            log.push(options.command);
            if (options.cwd) {
                log.push(`(cwd: ${options.cwd})`);
            }
            this.log(log.join(' '));
        }
        const cwd = (_c = options.cwd) !== null && _c !== void 0 ? _c : this.workdir;
        if (!fs_1.existsSync(cwd) || !fs_1.statSync(cwd).isDirectory()) {
            throw new Error(`invalid workdir (cwd): ${cwd} must be an existing directory`);
        }
        return child_process_1.spawnSync(options.command, {
            ...options,
            cwd,
            shell: true,
            stdio: 'inherit',
            env: {
                ...process.env,
                ...this.env,
            },
            ...options.spawnOptions,
        });
    }
    shellEval(options) {
        return this.shell({
            quiet: true,
            ...options,
            spawnOptions: {
                stdio: ['inherit', 'pipe', 'inherit'],
            },
        });
    }
    renderBuiltin(builtin) {
        const moduleRoot = path_1.dirname(require.resolve('../package.json'));
        const program = require.resolve(path_1.join(moduleRoot, 'lib', `${builtin}.task.js`));
        return `${process.execPath} ${program}`;
    }
}
//# sourceMappingURL=data:application/json;base64,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