"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Eslint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const common_1 = require("../common");
const component_1 = require("../component");
const json_1 = require("../json");
/**
 * (experimental) Represents eslint configuration.
 *
 * @experimental
 */
class Eslint extends component_1.Component {
    /**
     * @experimental
     */
    constructor(project, options) {
        var _b, _c, _d, _e, _f, _g;
        super(project);
        project.addDevDeps('eslint@^8', '@typescript-eslint/eslint-plugin@^5', '@typescript-eslint/parser@^5', 'eslint-import-resolver-node', 'eslint-import-resolver-typescript', 'eslint-plugin-import', 'json-schema');
        if (options.prettier) {
            project.addDevDeps('prettier', 'eslint-plugin-prettier', 'eslint-config-prettier');
        }
        if (options.aliasMap) {
            project.addDevDeps('eslint-import-resolver-alias');
        }
        const devdirs = (_b = options.devdirs) !== null && _b !== void 0 ? _b : [];
        const dirs = [...options.dirs, ...devdirs];
        const fileExtensions = (_c = options.fileExtensions) !== null && _c !== void 0 ? _c : ['.ts'];
        this._allowDevDeps = new Set((devdirs !== null && devdirs !== void 0 ? devdirs : []).map(dir => `**/${dir}/**`));
        const lintProjenRc = (_d = options.lintProjenRc) !== null && _d !== void 0 ? _d : true;
        const eslint = project.addTask('eslint', {
            description: 'Runs eslint against the codebase',
            exec: [
                'eslint',
                `--ext ${fileExtensions.join(',')}`,
                '--fix',
                '--no-error-on-unmatched-pattern',
                ...dirs,
                ...lintProjenRc ? [common_1.PROJEN_RC] : [],
            ].join(' '),
        });
        project.testTask.spawn(eslint);
        // exclude some files
        (_e = project.npmignore) === null || _e === void 0 ? void 0 : _e.exclude('/.eslintrc.json');
        const formattingRules = options.prettier ? {
            'prettier/prettier': ['error'],
        } : {
            // see https://github.com/typescript-eslint/typescript-eslint/blob/master/packages/eslint-plugin/docs/rules/indent.md
            'indent': ['off'],
            '@typescript-eslint/indent': ['error', 2],
            // Style
            'quotes': ['error', 'single', { avoidEscape: true }],
            'comma-dangle': ['error', 'always-multiline'],
            'comma-spacing': ['error', { before: false, after: true }],
            'no-multi-spaces': ['error', { ignoreEOLComments: false }],
            'array-bracket-spacing': ['error', 'never'],
            'array-bracket-newline': ['error', 'consistent'],
            'object-curly-spacing': ['error', 'always'],
            'object-curly-newline': ['error', { multiline: true, consistent: true }],
            'object-property-newline': ['error', { allowAllPropertiesOnSameLine: true }],
            'keyword-spacing': ['error'],
            'brace-style': ['error', '1tbs', { allowSingleLine: true }],
            'space-before-blocks': ['error'],
            'curly': ['error', 'multi-line', 'consistent'],
            '@typescript-eslint/member-delimiter-style': ['error'],
            // Require semicolons
            'semi': ['error', 'always'],
            // Max line lengths
            'max-len': ['error', {
                    code: 150,
                    ignoreUrls: true,
                    ignoreStrings: true,
                    ignoreTemplateLiterals: true,
                    ignoreComments: true,
                    ignoreRegExpLiterals: true,
                }],
            // Don't unnecessarily quote properties
            'quote-props': ['error', 'consistent-as-needed'],
        };
        this.rules = {
            ...formattingRules,
            // Require use of the `import { foo } from 'bar';` form instead of `import foo = require('bar');`
            '@typescript-eslint/no-require-imports': ['error'],
            // Require all imported dependencies are actually declared in package.json
            'import/no-extraneous-dependencies': [
                'error',
                {
                    // Only allow importing devDependencies from "devdirs".
                    devDependencies: () => this.renderDevDepsAllowList(),
                    optionalDependencies: false,
                    peerDependencies: true,
                },
            ],
            // Require all imported libraries actually resolve (!!required for import/no-extraneous-dependencies to work!!)
            'import/no-unresolved': ['error'],
            // Require an ordering on all imports
            'import/order': ['warn', {
                    groups: ['builtin', 'external'],
                    alphabetize: { order: 'asc', caseInsensitive: true },
                }],
            // Cannot import from the same module twice
            'no-duplicate-imports': ['error'],
            // Cannot shadow names
            'no-shadow': ['off'],
            '@typescript-eslint/no-shadow': ['error'],
            // Required spacing in property declarations (copied from TSLint, defaults are good)
            'key-spacing': ['error'],
            // No multiple empty lines
            'no-multiple-empty-lines': ['error'],
            // One of the easiest mistakes to make
            '@typescript-eslint/no-floating-promises': ['error'],
            // Make sure that inside try/catch blocks, promises are 'return await'ed
            // (must disable the base rule as it can report incorrect errors)
            'no-return-await': ['off'],
            '@typescript-eslint/return-await': ['error'],
            // Useless diff results
            'no-trailing-spaces': ['error'],
            // Must use foo.bar instead of foo['bar'] if possible
            'dot-notation': ['error'],
            // Are you sure | is not a typo for || ?
            'no-bitwise': ['error'],
            // Member ordering
            '@typescript-eslint/member-ordering': ['error', {
                    default: [
                        'public-static-field',
                        'public-static-method',
                        'protected-static-field',
                        'protected-static-method',
                        'private-static-field',
                        'private-static-method',
                        'field',
                        // Constructors
                        'constructor',
                        // Methods
                        'method',
                    ],
                }],
        };
        // Overrides for .projenrc.js
        this.overrides = [
            {
                files: [common_1.PROJEN_RC],
                rules: {
                    '@typescript-eslint/no-require-imports': 'off',
                    'import/no-extraneous-dependencies': 'off',
                },
            },
        ];
        this.ignorePatterns = (_f = options.ignorePatterns) !== null && _f !== void 0 ? _f : [
            '*.js',
            `!${common_1.PROJEN_RC}`,
            '*.d.ts',
            'node_modules/',
            '*.generated.ts',
            'coverage',
        ];
        const tsconfig = (_g = options.tsconfigPath) !== null && _g !== void 0 ? _g : './tsconfig.json';
        const plugins = [
            '@typescript-eslint',
            'import',
            ...(options.prettier ? ['prettier'] : []),
        ];
        const extendsConf = [
            'plugin:import/typescript',
            ...(options.prettier ? [
                'prettier',
                'plugin:prettier/recommended',
            ] : []),
        ];
        this.config = {
            env: {
                jest: true,
                node: true,
            },
            root: true,
            plugins,
            parser: '@typescript-eslint/parser',
            parserOptions: {
                ecmaVersion: 2018,
                sourceType: 'module',
                project: tsconfig,
            },
            extends: extendsConf,
            settings: {
                'import/parsers': {
                    '@typescript-eslint/parser': ['.ts', '.tsx'],
                },
                'import/resolver': {
                    ...(options.aliasMap && {
                        alias: {
                            map: Object.entries(options.aliasMap).map(([k, v]) => [k, v]),
                            extensions: options.aliasExtensions,
                        },
                    }),
                    node: {},
                    typescript: {
                        project: tsconfig,
                        ...(options.tsAlwaysTryTypes !== false && { alwaysTryTypes: true }),
                    },
                },
            },
            ignorePatterns: this.ignorePatterns,
            rules: this.rules,
            overrides: this.overrides,
        };
        new json_1.JsonFile(project, '.eslintrc.json', { obj: this.config, marker: false });
    }
    /**
     * (experimental) Returns the singletone Eslint component of a project or undefined if there is none.
     *
     * @experimental
     */
    static of(project) {
        const isEslint = (c) => c instanceof Eslint;
        return project.components.find(isEslint);
    }
    /**
     * (experimental) Add an eslint rule.
     *
     * @experimental
     */
    addRules(rules) {
        for (const [k, v] of Object.entries(rules)) {
            this.rules[k] = v;
        }
    }
    /**
     * (experimental) Add an eslint override.
     *
     * @experimental
     */
    addOverride(override) {
        this.overrides.push(override);
    }
    /**
     * (experimental) Do not lint these files.
     *
     * @experimental
     */
    addIgnorePattern(pattern) {
        this.ignorePatterns.push(pattern);
    }
    /**
     * (experimental) Add a glob file pattern which allows importing dev dependencies.
     *
     * @param pattern glob pattern.
     * @experimental
     */
    allowDevDeps(pattern) {
        this._allowDevDeps.add(pattern);
    }
    renderDevDepsAllowList() {
        return Array.from(this._allowDevDeps);
    }
}
exports.Eslint = Eslint;
_a = JSII_RTTI_SYMBOL_1;
Eslint[_a] = { fqn: "projen.javascript.Eslint", version: "0.40.2" };
//# sourceMappingURL=data:application/json;base64,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