"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Pom = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const semver_1 = require("../util/semver");
const xmlfile_1 = require("../xmlfile");
const POM_XML_ATTRS = {
    '@xsi:schemaLocation': 'http://maven.apache.org/POM/4.0.0 http://maven.apache.org/xsd/maven-4.0.0.xsd',
    '@xmlns': 'http://maven.apache.org/POM/4.0.0',
    '@xmlns:xsi': 'http://www.w3.org/2001/XMLSchema-instance',
};
/**
 * (experimental) A Project Object Model or POM is the fundamental unit of work in Maven.
 *
 * It is
 * an XML file that contains information about the project and configuration
 * details used by Maven to build the project.
 *
 * @experimental
 */
class Pom extends component_1.Component {
    /**
     * @experimental
     */
    constructor(project, options) {
        var _b;
        super(project);
        this.properties = {};
        this.fileName = 'pom.xml';
        this.groupId = options.groupId;
        this.artifactId = options.artifactId;
        this.version = options.version;
        this.packaging = (_b = options.packaging) !== null && _b !== void 0 ? _b : 'jar';
        this.name = project.name;
        this.description = options.description;
        this.url = options.url;
        new xmlfile_1.XmlFile(project, this.fileName, { obj: () => this.synthPom() });
    }
    /**
     * (experimental) Adds a key/value property to the pom.
     *
     * @param key the key.
     * @param value the value.
     * @experimental
     */
    addProperty(key, value) {
        this.properties[key] = value;
    }
    /**
     * (experimental) Adds a runtime dependency.
     *
     * @param spec Format `<groupId>/<artifactId>@<semver>`.
     * @experimental
     */
    addDependency(spec) {
        this.project.deps.addDependency(spec, dependencies_1.DependencyType.RUNTIME);
    }
    /**
     * (experimental) Adds a test dependency.
     *
     * @param spec Format `<groupId>/<artifactId>@<semver>`.
     * @experimental
     */
    addTestDependency(spec) {
        this.project.deps.addDependency(spec, dependencies_1.DependencyType.TEST);
    }
    /**
     * (experimental) Adds a build plugin to the pom.
     *
     * The plug in is also added as a BUILD dep to the project.
     *
     * @param spec dependency spec (`group/artifact@version`).
     * @param options plugin options.
     * @experimental
     */
    addPlugin(spec, options = {}) {
        var _b;
        for (const dep of (_b = options.dependencies) !== null && _b !== void 0 ? _b : []) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUILD);
        }
        return this.project.deps.addDependency(spec, dependencies_1.DependencyType.BUILD, options);
    }
    synthPom() {
        return _resolve_1.resolve({
            project: {
                ...POM_XML_ATTRS,
                modelVersion: '4.0.0',
                groupId: this.groupId,
                artifactId: this.artifactId,
                version: this.version,
                packaging: this.packaging,
                name: this.name,
                description: this.description,
                url: this.url,
                properties: this.properties,
                ...this.synthDependencies(),
            },
        }, { omitEmpty: true });
    }
    synthDependencies() {
        const deps = this.project.deps.all;
        if (deps.length === 0) {
            return;
        }
        const dependencies = [];
        const plugins = [];
        for (const dep of deps) {
            switch (dep.type) {
                case dependencies_1.DependencyType.PEER:
                case dependencies_1.DependencyType.RUNTIME:
                    dependencies.push(mavenCoords(dep));
                    break;
                case dependencies_1.DependencyType.TEST:
                    dependencies.push({
                        ...mavenCoords(dep),
                        scope: 'test',
                    });
                    break;
                // build maps to plugins
                case dependencies_1.DependencyType.BUILD:
                    plugins.push({
                        ...mavenCoords(dep),
                        ...pluginConfig(dep.metadata),
                    });
                    break;
                default:
                    throw new Error(`unsupported dependency type: ${dep.type}`);
            }
        }
        return {
            build: { plugins: { plugin: plugins } },
            dependencies: { dependency: dependencies },
        };
    }
}
exports.Pom = Pom;
_a = JSII_RTTI_SYMBOL_1;
Pom[_a] = { fqn: "projen.java.Pom", version: "0.40.2" };
/**
 * Parses maven groupId and artifactId from a dependency name.
 *
 *     name    <=> <groupId>/<artifactId>
 *     version <=> <version>
 */
function mavenCoords(dep) {
    const name = dep.name;
    const parts = name.split('/');
    if (parts.length !== 2) {
        throw new Error(`invalid maven coordinates in dependency named "${name}". format is "<groupId>/<artifactId>". For example "org.junit.jupiter/junit-jupiter-engine"`);
    }
    return {
        groupId: parts[0],
        artifactId: parts[1],
        version: dep.version ? semver_1.toMavenVersionRange(dep.version) : undefined,
    };
}
function pluginConfig(options = {}) {
    var _b, _c;
    return {
        configuration: options.configuration,
        dependencies: (options.dependencies && options.dependencies.length > 0)
            ? { dependency: (_b = options.dependencies) === null || _b === void 0 ? void 0 : _b.map(d => mavenCoords(dependencies_1.Dependencies.parseDependency(d))) }
            : undefined,
        executions: (_c = options.executions) === null || _c === void 0 ? void 0 : _c.map(e => ({
            execution: {
                id: e.id,
                goals: e.goals.map(goal => ({ goal })),
            },
        })),
    };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicG9tLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2phdmEvcG9tLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsMENBQXNDO0FBQ3RDLDRDQUF5QztBQUN6QyxrREFBc0Y7QUFFdEYsMkNBQXFEO0FBQ3JELHdDQUFxQztBQUVyQyxNQUFNLGFBQWEsR0FBRztJQUNwQixxQkFBcUIsRUFBRSwrRUFBK0U7SUFDdEcsUUFBUSxFQUFFLG1DQUFtQztJQUM3QyxZQUFZLEVBQUUsMkNBQTJDO0NBQzFELENBQUM7Ozs7Ozs7Ozs7QUF3QkYsTUFBYSxHQUFJLFNBQVEscUJBQVM7Ozs7SUEyQmhDLFlBQVksT0FBZ0IsRUFBRSxPQUFtQjs7UUFDL0MsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBSEEsZUFBVSxHQUF3QixFQUFFLENBQUM7UUFLcEQsSUFBSSxDQUFDLFFBQVEsR0FBRyxTQUFTLENBQUM7UUFDMUIsSUFBSSxDQUFDLE9BQU8sR0FBRyxPQUFPLENBQUMsT0FBTyxDQUFDO1FBQy9CLElBQUksQ0FBQyxVQUFVLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztRQUNyQyxJQUFJLENBQUMsT0FBTyxHQUFHLE9BQU8sQ0FBQyxPQUFPLENBQUM7UUFDL0IsSUFBSSxDQUFDLFNBQVMsU0FBRyxPQUFPLENBQUMsU0FBUyxtQ0FBSSxLQUFLLENBQUM7UUFDNUMsSUFBSSxDQUFDLElBQUksR0FBRyxPQUFPLENBQUMsSUFBSSxDQUFDO1FBQ3pCLElBQUksQ0FBQyxXQUFXLEdBQUcsT0FBTyxDQUFDLFdBQVcsQ0FBQztRQUN2QyxJQUFJLENBQUMsR0FBRyxHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUM7UUFFdkIsSUFBSSxpQkFBTyxDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsUUFBUSxFQUFFLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDdEUsQ0FBQzs7Ozs7Ozs7SUFHTSxXQUFXLENBQUMsR0FBVyxFQUFFLEtBQWE7UUFDM0MsSUFBSSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUM7SUFDL0IsQ0FBQzs7Ozs7OztJQUdNLGFBQWEsQ0FBQyxJQUFZO1FBQy9CLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsNkJBQWMsQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUNoRSxDQUFDOzs7Ozs7O0lBR00saUJBQWlCLENBQUMsSUFBWTtRQUNuQyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLDZCQUFjLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDN0QsQ0FBQzs7Ozs7Ozs7OztJQUdNLFNBQVMsQ0FBQyxJQUFZLEVBQUUsVUFBeUIsRUFBRTs7UUFDeEQsS0FBSyxNQUFNLEdBQUcsVUFBSSxPQUFPLENBQUMsWUFBWSxtQ0FBSSxFQUFFLEVBQUU7WUFDNUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEdBQUcsRUFBRSw2QkFBYyxDQUFDLEtBQUssQ0FBQyxDQUFDO1NBQzVEO1FBQ0QsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLDZCQUFjLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQzlFLENBQUM7SUFFTyxRQUFRO1FBQ2QsT0FBTyxrQkFBTyxDQUFDO1lBQ2IsT0FBTyxFQUFFO2dCQUNQLEdBQUcsYUFBYTtnQkFDaEIsWUFBWSxFQUFFLE9BQU87Z0JBQ3JCLE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTztnQkFDckIsVUFBVSxFQUFFLElBQUksQ0FBQyxVQUFVO2dCQUMzQixPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU87Z0JBQ3JCLFNBQVMsRUFBRSxJQUFJLENBQUMsU0FBUztnQkFDekIsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJO2dCQUNmLFdBQVcsRUFBRSxJQUFJLENBQUMsV0FBVztnQkFDN0IsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO2dCQUNiLFVBQVUsRUFBRSxJQUFJLENBQUMsVUFBVTtnQkFDM0IsR0FBRyxJQUFJLENBQUMsaUJBQWlCLEVBQUU7YUFDNUI7U0FDRixFQUFFLEVBQUUsU0FBUyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7SUFDMUIsQ0FBQztJQUVPLGlCQUFpQjtRQUN2QixNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUM7UUFDbkMsSUFBSSxJQUFJLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUFFLE9BQU87U0FBRTtRQUVsQyxNQUFNLFlBQVksR0FBVSxFQUFFLENBQUM7UUFDL0IsTUFBTSxPQUFPLEdBQVUsRUFBRSxDQUFDO1FBRTFCLEtBQUssTUFBTSxHQUFHLElBQUksSUFBSSxFQUFFO1lBQ3RCLFFBQVEsR0FBRyxDQUFDLElBQUksRUFBRTtnQkFDaEIsS0FBSyw2QkFBYyxDQUFDLElBQUksQ0FBQztnQkFDekIsS0FBSyw2QkFBYyxDQUFDLE9BQU87b0JBQ3pCLFlBQVksQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7b0JBQ3BDLE1BQU07Z0JBRVIsS0FBSyw2QkFBYyxDQUFDLElBQUk7b0JBQ3RCLFlBQVksQ0FBQyxJQUFJLENBQUM7d0JBQ2hCLEdBQUcsV0FBVyxDQUFDLEdBQUcsQ0FBQzt3QkFDbkIsS0FBSyxFQUFFLE1BQU07cUJBQ2QsQ0FBQyxDQUFDO29CQUNILE1BQU07Z0JBRVIsd0JBQXdCO2dCQUN4QixLQUFLLDZCQUFjLENBQUMsS0FBSztvQkFDdkIsT0FBTyxDQUFDLElBQUksQ0FBQzt3QkFDWCxHQUFHLFdBQVcsQ0FBQyxHQUFHLENBQUM7d0JBQ25CLEdBQUcsWUFBWSxDQUFDLEdBQUcsQ0FBQyxRQUF5QixDQUFDO3FCQUMvQyxDQUFDLENBQUM7b0JBQ0gsTUFBTTtnQkFFUjtvQkFDRSxNQUFNLElBQUksS0FBSyxDQUFDLGdDQUFnQyxHQUFHLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQzthQUMvRDtTQUNGO1FBRUQsT0FBTztZQUNMLEtBQUssRUFBRSxFQUFFLE9BQU8sRUFBRSxFQUFFLE1BQU0sRUFBRSxPQUFPLEVBQUUsRUFBRTtZQUN2QyxZQUFZLEVBQUUsRUFBRSxVQUFVLEVBQUUsWUFBWSxFQUFFO1NBQzNDLENBQUM7SUFDSixDQUFDOztBQXpISCxrQkEwSEM7OztBQXVCRDs7Ozs7R0FLRztBQUNILFNBQVMsV0FBVyxDQUFDLEdBQTBCO0lBQzdDLE1BQU0sSUFBSSxHQUFHLEdBQUcsQ0FBQyxJQUFJLENBQUM7SUFDdEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUM5QixJQUFJLEtBQUssQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1FBQ3RCLE1BQU0sSUFBSSxLQUFLLENBQUMsa0RBQWtELElBQUksNkZBQTZGLENBQUMsQ0FBQztLQUN0SztJQUVELE9BQU87UUFDTCxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUMsQ0FBQztRQUNqQixVQUFVLEVBQUUsS0FBSyxDQUFDLENBQUMsQ0FBQztRQUNwQixPQUFPLEVBQUUsR0FBRyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsNEJBQW1CLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO0tBQ3BFLENBQUM7QUFDSixDQUFDO0FBRUQsU0FBUyxZQUFZLENBQUMsVUFBeUIsRUFBRTs7SUFDL0MsT0FBTztRQUNMLGFBQWEsRUFBRSxPQUFPLENBQUMsYUFBYTtRQUNwQyxZQUFZLEVBQUUsQ0FBQyxPQUFPLENBQUMsWUFBWSxJQUFJLE9BQU8sQ0FBQyxZQUFZLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQztZQUNyRSxDQUFDLENBQUMsRUFBRSxVQUFVLFFBQUUsT0FBTyxDQUFDLFlBQVksMENBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsV0FBVyxDQUFDLDJCQUFZLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUM5RixDQUFDLENBQUMsU0FBUztRQUNiLFVBQVUsUUFBRSxPQUFPLENBQUMsVUFBVSwwQ0FBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1lBQ3hDLFNBQVMsRUFBRTtnQkFDVCxFQUFFLEVBQUUsQ0FBQyxDQUFDLEVBQUU7Z0JBQ1IsS0FBSyxFQUFFLENBQUMsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7YUFDdkM7U0FDRixDQUFDLENBQUM7S0FDSixDQUFDO0FBQ0osQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IHJlc29sdmUgfSBmcm9tICcuLi9fcmVzb2x2ZSc7XG5pbXBvcnQgeyBDb21wb25lbnQgfSBmcm9tICcuLi9jb21wb25lbnQnO1xuaW1wb3J0IHsgRGVwZW5kZW5jaWVzLCBEZXBlbmRlbmN5Q29vcmRpbmF0ZXMsIERlcGVuZGVuY3lUeXBlIH0gZnJvbSAnLi4vZGVwZW5kZW5jaWVzJztcbmltcG9ydCB7IFByb2plY3QgfSBmcm9tICcuLi9wcm9qZWN0JztcbmltcG9ydCB7IHRvTWF2ZW5WZXJzaW9uUmFuZ2UgfSBmcm9tICcuLi91dGlsL3NlbXZlcic7XG5pbXBvcnQgeyBYbWxGaWxlIH0gZnJvbSAnLi4veG1sZmlsZSc7XG5cbmNvbnN0IFBPTV9YTUxfQVRUUlMgPSB7XG4gICdAeHNpOnNjaGVtYUxvY2F0aW9uJzogJ2h0dHA6Ly9tYXZlbi5hcGFjaGUub3JnL1BPTS80LjAuMCBodHRwOi8vbWF2ZW4uYXBhY2hlLm9yZy94c2QvbWF2ZW4tNC4wLjAueHNkJyxcbiAgJ0B4bWxucyc6ICdodHRwOi8vbWF2ZW4uYXBhY2hlLm9yZy9QT00vNC4wLjAnLFxuICAnQHhtbG5zOnhzaSc6ICdodHRwOi8vd3d3LnczLm9yZy8yMDAxL1hNTFNjaGVtYS1pbnN0YW5jZScsXG59O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBQb21PcHRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBncm91cElkOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGFydGlmYWN0SWQ6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2ZXJzaW9uOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcGFja2FnaW5nPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkZXNjcmlwdGlvbj86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHVybD86IHN0cmluZztcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBQb20gZXh0ZW5kcyBDb21wb25lbnQge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBmaWxlTmFtZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgZ3JvdXBJZDogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgYXJ0aWZhY3RJZDogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHZlcnNpb246IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgcGFja2FnaW5nOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBuYW1lPzogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBkZXNjcmlwdGlvbj86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHVybD86IHN0cmluZztcblxuICBwcml2YXRlIHJlYWRvbmx5IHByb3BlcnRpZXM6IFJlY29yZDxzdHJpbmcsIGFueT4gPSB7fTtcblxuICBjb25zdHJ1Y3Rvcihwcm9qZWN0OiBQcm9qZWN0LCBvcHRpb25zOiBQb21PcHRpb25zKSB7XG4gICAgc3VwZXIocHJvamVjdCk7XG5cbiAgICB0aGlzLmZpbGVOYW1lID0gJ3BvbS54bWwnO1xuICAgIHRoaXMuZ3JvdXBJZCA9IG9wdGlvbnMuZ3JvdXBJZDtcbiAgICB0aGlzLmFydGlmYWN0SWQgPSBvcHRpb25zLmFydGlmYWN0SWQ7XG4gICAgdGhpcy52ZXJzaW9uID0gb3B0aW9ucy52ZXJzaW9uO1xuICAgIHRoaXMucGFja2FnaW5nID0gb3B0aW9ucy5wYWNrYWdpbmcgPz8gJ2phcic7XG4gICAgdGhpcy5uYW1lID0gcHJvamVjdC5uYW1lO1xuICAgIHRoaXMuZGVzY3JpcHRpb24gPSBvcHRpb25zLmRlc2NyaXB0aW9uO1xuICAgIHRoaXMudXJsID0gb3B0aW9ucy51cmw7XG5cbiAgICBuZXcgWG1sRmlsZShwcm9qZWN0LCB0aGlzLmZpbGVOYW1lLCB7IG9iajogKCkgPT4gdGhpcy5zeW50aFBvbSgpIH0pO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFkZFByb3BlcnR5KGtleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nKSB7XG4gICAgdGhpcy5wcm9wZXJ0aWVzW2tleV0gPSB2YWx1ZTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhZGREZXBlbmRlbmN5KHNwZWM6IHN0cmluZykge1xuICAgIHRoaXMucHJvamVjdC5kZXBzLmFkZERlcGVuZGVuY3koc3BlYywgRGVwZW5kZW5jeVR5cGUuUlVOVElNRSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkVGVzdERlcGVuZGVuY3koc3BlYzogc3RyaW5nKSB7XG4gICAgdGhpcy5wcm9qZWN0LmRlcHMuYWRkRGVwZW5kZW5jeShzcGVjLCBEZXBlbmRlbmN5VHlwZS5URVNUKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgYWRkUGx1Z2luKHNwZWM6IHN0cmluZywgb3B0aW9uczogUGx1Z2luT3B0aW9ucyA9IHt9KSB7XG4gICAgZm9yIChjb25zdCBkZXAgb2Ygb3B0aW9ucy5kZXBlbmRlbmNpZXMgPz8gW10pIHtcbiAgICAgIHRoaXMucHJvamVjdC5kZXBzLmFkZERlcGVuZGVuY3koZGVwLCBEZXBlbmRlbmN5VHlwZS5CVUlMRCk7XG4gICAgfVxuICAgIHJldHVybiB0aGlzLnByb2plY3QuZGVwcy5hZGREZXBlbmRlbmN5KHNwZWMsIERlcGVuZGVuY3lUeXBlLkJVSUxELCBvcHRpb25zKTtcbiAgfVxuXG4gIHByaXZhdGUgc3ludGhQb20oKSB7XG4gICAgcmV0dXJuIHJlc29sdmUoe1xuICAgICAgcHJvamVjdDoge1xuICAgICAgICAuLi5QT01fWE1MX0FUVFJTLFxuICAgICAgICBtb2RlbFZlcnNpb246ICc0LjAuMCcsXG4gICAgICAgIGdyb3VwSWQ6IHRoaXMuZ3JvdXBJZCxcbiAgICAgICAgYXJ0aWZhY3RJZDogdGhpcy5hcnRpZmFjdElkLFxuICAgICAgICB2ZXJzaW9uOiB0aGlzLnZlcnNpb24sXG4gICAgICAgIHBhY2thZ2luZzogdGhpcy5wYWNrYWdpbmcsXG4gICAgICAgIG5hbWU6IHRoaXMubmFtZSxcbiAgICAgICAgZGVzY3JpcHRpb246IHRoaXMuZGVzY3JpcHRpb24sXG4gICAgICAgIHVybDogdGhpcy51cmwsXG4gICAgICAgIHByb3BlcnRpZXM6IHRoaXMucHJvcGVydGllcyxcbiAgICAgICAgLi4udGhpcy5zeW50aERlcGVuZGVuY2llcygpLFxuICAgICAgfSxcbiAgICB9LCB7IG9taXRFbXB0eTogdHJ1ZSB9KTtcbiAgfVxuXG4gIHByaXZhdGUgc3ludGhEZXBlbmRlbmNpZXMoKSB7XG4gICAgY29uc3QgZGVwcyA9IHRoaXMucHJvamVjdC5kZXBzLmFsbDtcbiAgICBpZiAoZGVwcy5sZW5ndGggPT09IDApIHsgcmV0dXJuOyB9XG5cbiAgICBjb25zdCBkZXBlbmRlbmNpZXM6IGFueVtdID0gW107XG4gICAgY29uc3QgcGx1Z2luczogYW55W10gPSBbXTtcblxuICAgIGZvciAoY29uc3QgZGVwIG9mIGRlcHMpIHtcbiAgICAgIHN3aXRjaCAoZGVwLnR5cGUpIHtcbiAgICAgICAgY2FzZSBEZXBlbmRlbmN5VHlwZS5QRUVSOlxuICAgICAgICBjYXNlIERlcGVuZGVuY3lUeXBlLlJVTlRJTUU6XG4gICAgICAgICAgZGVwZW5kZW5jaWVzLnB1c2gobWF2ZW5Db29yZHMoZGVwKSk7XG4gICAgICAgICAgYnJlYWs7XG5cbiAgICAgICAgY2FzZSBEZXBlbmRlbmN5VHlwZS5URVNUOlxuICAgICAgICAgIGRlcGVuZGVuY2llcy5wdXNoKHtcbiAgICAgICAgICAgIC4uLm1hdmVuQ29vcmRzKGRlcCksXG4gICAgICAgICAgICBzY29wZTogJ3Rlc3QnLFxuICAgICAgICAgIH0pO1xuICAgICAgICAgIGJyZWFrO1xuXG4gICAgICAgIC8vIGJ1aWxkIG1hcHMgdG8gcGx1Z2luc1xuICAgICAgICBjYXNlIERlcGVuZGVuY3lUeXBlLkJVSUxEOlxuICAgICAgICAgIHBsdWdpbnMucHVzaCh7XG4gICAgICAgICAgICAuLi5tYXZlbkNvb3JkcyhkZXApLFxuICAgICAgICAgICAgLi4ucGx1Z2luQ29uZmlnKGRlcC5tZXRhZGF0YSBhcyBQbHVnaW5PcHRpb25zKSxcbiAgICAgICAgICB9KTtcbiAgICAgICAgICBicmVhaztcblxuICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgdW5zdXBwb3J0ZWQgZGVwZW5kZW5jeSB0eXBlOiAke2RlcC50eXBlfWApO1xuICAgICAgfVxuICAgIH1cblxuICAgIHJldHVybiB7XG4gICAgICBidWlsZDogeyBwbHVnaW5zOiB7IHBsdWdpbjogcGx1Z2lucyB9IH0sXG4gICAgICBkZXBlbmRlbmNpZXM6IHsgZGVwZW5kZW5jeTogZGVwZW5kZW5jaWVzIH0sXG4gICAgfTtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFBsdWdpbk9wdGlvbnMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjb25maWd1cmF0aW9uPzogeyBba2V5OiBzdHJpbmddOiBhbnkgfTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGV4ZWN1dGlvbnM/OiBQbHVnaW5FeGVjdXRpb25bXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkZXBlbmRlbmNpZXM/OiBzdHJpbmdbXTtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFBsdWdpbkV4ZWN1dGlvbiB7XG4gICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaWQ6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBnb2Fsczogc3RyaW5nW107XG59XG5cbi8qKlxuICogUGFyc2VzIG1hdmVuIGdyb3VwSWQgYW5kIGFydGlmYWN0SWQgZnJvbSBhIGRlcGVuZGVuY3kgbmFtZS5cbiAqXG4gKiAgICAgbmFtZSAgICA8PT4gPGdyb3VwSWQ+LzxhcnRpZmFjdElkPlxuICogICAgIHZlcnNpb24gPD0+IDx2ZXJzaW9uPlxuICovXG5mdW5jdGlvbiBtYXZlbkNvb3JkcyhkZXA6IERlcGVuZGVuY3lDb29yZGluYXRlcykge1xuICBjb25zdCBuYW1lID0gZGVwLm5hbWU7XG4gIGNvbnN0IHBhcnRzID0gbmFtZS5zcGxpdCgnLycpO1xuICBpZiAocGFydHMubGVuZ3RoICE9PSAyKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGBpbnZhbGlkIG1hdmVuIGNvb3JkaW5hdGVzIGluIGRlcGVuZGVuY3kgbmFtZWQgXCIke25hbWV9XCIuIGZvcm1hdCBpcyBcIjxncm91cElkPi88YXJ0aWZhY3RJZD5cIi4gRm9yIGV4YW1wbGUgXCJvcmcuanVuaXQuanVwaXRlci9qdW5pdC1qdXBpdGVyLWVuZ2luZVwiYCk7XG4gIH1cblxuICByZXR1cm4ge1xuICAgIGdyb3VwSWQ6IHBhcnRzWzBdLFxuICAgIGFydGlmYWN0SWQ6IHBhcnRzWzFdLFxuICAgIHZlcnNpb246IGRlcC52ZXJzaW9uID8gdG9NYXZlblZlcnNpb25SYW5nZShkZXAudmVyc2lvbikgOiB1bmRlZmluZWQsXG4gIH07XG59XG5cbmZ1bmN0aW9uIHBsdWdpbkNvbmZpZyhvcHRpb25zOiBQbHVnaW5PcHRpb25zID0ge30pIHtcbiAgcmV0dXJuIHtcbiAgICBjb25maWd1cmF0aW9uOiBvcHRpb25zLmNvbmZpZ3VyYXRpb24sXG4gICAgZGVwZW5kZW5jaWVzOiAob3B0aW9ucy5kZXBlbmRlbmNpZXMgJiYgb3B0aW9ucy5kZXBlbmRlbmNpZXMubGVuZ3RoID4gMClcbiAgICAgID8geyBkZXBlbmRlbmN5OiBvcHRpb25zLmRlcGVuZGVuY2llcz8ubWFwKGQgPT4gbWF2ZW5Db29yZHMoRGVwZW5kZW5jaWVzLnBhcnNlRGVwZW5kZW5jeShkKSkpIH1cbiAgICAgIDogdW5kZWZpbmVkLFxuICAgIGV4ZWN1dGlvbnM6IG9wdGlvbnMuZXhlY3V0aW9ucz8ubWFwKGUgPT4gKHtcbiAgICAgIGV4ZWN1dGlvbjoge1xuICAgICAgICBpZDogZS5pZCxcbiAgICAgICAgZ29hbHM6IGUuZ29hbHMubWFwKGdvYWwgPT4gKHsgZ29hbCB9KSksXG4gICAgICB9LFxuICAgIH0pKSxcbiAgfTtcbn1cbiJdfQ==