"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderInstallCommand = exports.installPackage = void 0;
const path = require("path");
const fs = require("fs-extra");
const logging = require("../logging");
const util_1 = require("../util");
/**
 * Installs the npm module (through `npm install`) to node_modules under `projectDir`.
 * @param spec The npm package spec (e.g. `foo@^1.2` or `foo@/var/folders/8k/qcw0ls5pv_ph0000gn/T/projen-RYurCw/pkg.tgz`)
 * @returns The installed package name (e.g. `@foo/bar`)
 */
function installPackage(baseDir, spec) {
    const packageJsonPath = path.join(baseDir, 'package.json');
    const packageJsonExisted = fs.existsSync(packageJsonPath);
    if (!packageJsonExisted) {
        // Make sure we have a package.json to read from later
        util_1.exec('npm init --yes', { cwd: baseDir });
    }
    logging.info(`installing external module ${spec}...`);
    util_1.exec(renderInstallCommand(baseDir, spec), { cwd: baseDir });
    // Get the true installed package name
    const packageJson = fs.readJsonSync(packageJsonPath);
    const packageName = Object.keys(packageJson.devDependencies).find(name => name !== 'projen');
    if (!packageName) {
        throw new Error(`Unable to resolve package name from spec ${spec}`);
    }
    // if package.json did not exist before calling `npm install`, we should remove it
    // so we can start off clean.
    if (!packageJsonExisted) {
        fs.removeSync(packageJsonPath);
    }
    return packageName;
}
exports.installPackage = installPackage;
/**
 * Render a command to install an npm package.
 *
 * Engine checks are ignorred at this point so that the module can be installed
 * regardless of the environment. This was needed to unblock the upgrade of the
 * minimum node version of projen, but also okay generally because engine checks
 * will be performed later and for all eternety.
 *
 * @param dir Base directory
 * @param module The module to install (e.g. foo@^1.2)
 * @returns The string that includes the install command ("npm install ...")
 */
function renderInstallCommand(dir, module) {
    return `npm install --save-dev -f --no-package-lock --prefix=${dir} ${module}`;
}
exports.renderInstallCommand = renderInstallCommand;
//# sourceMappingURL=data:application/json;base64,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