"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
const yargs = require("yargs");
const common_1 = require("../common");
const task_runtime_1 = require("../task-runtime");
const synth_1 = require("./synth");
const tasks_1 = require("./tasks");
const DEFAULT_RC = path_1.resolve(common_1.PROJEN_RC);
async function main() {
    const ya = yargs;
    ya.commandDir('cmds');
    const runtime = new task_runtime_1.TaskRuntime('.');
    tasks_1.discoverTaskCommands(runtime, ya);
    ya.recommendCommands();
    ya.strictCommands();
    ya.showHelpOnFail(false);
    ya.wrap(yargs.terminalWidth());
    ya.option('post', { type: 'boolean', default: true, desc: 'Run post-synthesis steps such as installing dependencies. Use --no-post to skip' });
    ya.option('watch', { type: 'boolean', default: false, desc: 'Keep running and resynthesize when projenrc changes', alias: 'w' });
    ya.options('debug', { type: 'boolean', default: false, desc: 'Debug logs' });
    ya.options('rc', { desc: 'path to .projenrc.js file', default: DEFAULT_RC, type: 'string' });
    ya.completion();
    ya.help();
    // do not use the default yargs '--version' implementation since it is
    // global by default (it appears on all subcommands)
    ya.version(false);
    ya.option('version', { type: 'boolean', description: 'Show version number', global: false });
    const args = ya.argv;
    if (args.debug) {
        process.env.DEBUG = 'true';
    }
    // no command means synthesize
    if (args._.length === 0) {
        if (args.version) {
            console.log(common_1.PROJEN_VERSION);
            process.exit(0);
        }
        await synth_1.synth(runtime, {
            post: args.post,
            watch: args.watch,
            rcfile: args.rc,
        });
    }
}
main().catch(e => {
    console.error(e.stack);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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