# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pyentrypoint']

package_data = \
{'': ['*']}

install_requires = \
['colorlog>=4.1,<5.0',
 'jinja2>=2.11,<3.0',
 'pyyaml>=5.3,<6.0',
 'six>=1.14,<2.0',
 'watchdog>=0.10,<0.11']

entry_points = \
{'console_scripts': ['pyentrypoint = pyentrypoint.__main__:main']}

setup_kwargs = {
    'name': 'pyentrypoint',
    'version': '0.7.1',
    'description': 'pyentrypoint manages entrypoints in Docker containers.',
    'long_description': '# pyentrypoint\n\n__pyentrypoint__ is a tool written in `Python` to manage Docker containers `ENTRYPOINT`.\n\nThis tool avoids writing shell scripts to:\n - Handle commands and sub commands\n - Identify linked containers\n - Generate configuration using `jinja2` templates\n - Run commands before starting service\n - Reload service when configuration has changed\n\n[![Documentation Status](https://readthedocs.org/projects/pyentrypoint/badge/?version=latest)](http://pyentrypoint.readthedocs.io/en/latest/?badge=latest) [![Build Status](https://travis-ci.org/cmehay/pyentrypoint.svg?branch=master)](https://travis-ci.org/cmehay/pyentrypoint)\n\n\n## Changelog\n\n###### v0.7.1 (2020-05-24)\n  - add envtobool function in configuration template\n\n###### v0.7.0 (2020-05-17)\n  - Add command matching setup\n\n###### V0.6.0 (2020-05-10)\n  - Drop python 2 support\n  - Deprecation of `command` and `subcommands` settings for `commands` (see bellow)\n\n## Usages\n\n### Install in container\n\nAll you need to do is to setup a `yaml` file called `entrypoint-config.yml` and to install __pyentrypoint__ in your `Dockerfile` using pip.\n\n```dockerfile\nFROM        debian\n# Installing git for example\nRUN         apt-get update && apt-get install git python-pip -y\n# Install pyentrypoint\nRUN         pip install pyentrypoint\n# Copy config file in the current WORKDIR\nCOPY        entrypoint-config.yml .\n# Set ENTRYPOINT\nENTRYPOINT  [\'pyentrypoint\']\n# git will be the default command\nCMD         [\'git\']\n```\n\n```dockerfile\nFROM        alpine\n# Installing git for example\nRUN         apk add --update py-pip git\n# Install pyentrypoint\nRUN         pip install pyentrypoint\n# Copy config file in the current WORKDIR\nCOPY        entrypoint-config.yml .\n# Set ENTRYPOINT\nENTRYPOINT  [\'pyentrypoint\']\n# git will be the default command\nCMD         [\'git\']\n```\n\n#### Using docker-image\n\n```dockerfile\nFROM    goldy/pyentrypoint:python3\n\n# ONBUILD statement add entrypoint-config.yml in current directories\n\n```\n\nAvailable with many flavours:\n\n- `goldy/pyentrypoint:python3`\n- `goldy/pyentrypoint:python3.6`\n- `goldy/pyentrypoint:python3.7`\n- `goldy/pyentrypoint:python3.8`\n- `goldy/pyentrypoint:python3-alpine`\n- `goldy/pyentrypoint:python3.6-alpine`\n- `goldy/pyentrypoint:python3.7-alpine`\n- `goldy/pyentrypoint:python3.8-alpine`\n\n\n\n\n### Working examples\n - [Tor hidden service](https://github.com/cmehay/docker-tor-hidden-service)\n\n### Setup entrypoint\n\nThis is an example of `entrypoint-config.yml` file.\n\n```yaml\n# Entrypoint configuration example\n\n# This setup lists commands handled by entrypoint.\n# If you run the container with a command not in this list,\n# pyentrypoint will run the command directly without any action\n# If this setting and `command` are not set, all commands will be handled.\n# Support wildcard\ncommands:\n    - git\n    - sl*\n\n# DEPRECATED: This setup is remplaced by `commands`\n# This entry should reflect CMD in Dockerfile\n# If `commands` is present, this setup will be ignored.\n# DEPRECATED: This setup is remplaced by `commands`\ncommand: git\n\n# DEPRECATED: This setup will be dropped\n# This is a list with some subcommands to handle\n# when CMD is not `git` here.\n# By default, all args started with hyphen are handled.\n# DEPRECATED: This setup will be dropped\nsubcommands:\n    - "-*"\n    - clone\n    - init\n    - ls-files\n    # etc...\n\n# User and group to run the cmd.\n# Can be name or uid/gid.\n# Affect only command handled.\n# Dockerfile USER value by default.\nuser: 1000\ngroup: 1000\n\n# These files should exist (ADD, COPY or mounted)\n# and should be jinja templated.\n# Note: if config files end with ".tpl", the extension will be removed.\nconfig_files:\n    - /etc/gitconfig\n    - .ssh/config.tpl # Will apply to ".ssh/config"\n    - /tmp/id_rsa: .ssh/id_rsa # Will apply "/tmp/id_rsa" template to ".ssh/id_rsa"\n\n\n# These environment variables will be wiped before\n# exec command to keep them secret\n# CAUTION: if the container is linked to another one,\n# theses variables will passed to it anyway\nsecret_env:\n    - SSHKEY\n    - \'*\' # Support globbing, all environment will be wiped\n\n# Links are handled here\n# Port, name, protocol or env variable can be used to identify the links\n# Raise an error if the link could not be identified\n# This is not supported when using docker network or docker-compose v2.\nlinks:\n    \'ssh\':\n        port: 22\n        name: \'ssh*\'\n        protocol: tcp\n        # env can be list, dictionary or string\n        env:\n            FOO: bar\n        # Single doesn\'t allow multiple links for this ID\n        # false by default\n        single: true\n        # Set to false to get optional link\n        # true by default\n        required: true\n\n# Commands to run before applying configuration\npre_conf_commands:\n    - echo something > to_this_file\n\n# commands to run after applying configuration\npost_conf_commands:\n    - echo "something else" > to_this_another_file\n\npost_run_commands:\n    - echo run commands after started service\n\n# Reload service when configuration change by sending a signal to process\nreload:\n    signal: SIGHUP # Optional, signal to send, default is SIGHUP\n    pid: 1 # Optional, pid to send signal, default is 1\n    watch_config_files: true # Optional, watch defined config files, default True\n    files: # Optional, list of files to watch\n        - /etc/conf/to/watch\n        - /file/support/*.matching\n# can also be enabled like this:\nreload: true\n\n\n# Cleanup environment from variables created by linked containers\n# before running command (True by default)\nclean_env: true\n\n# Enable debug to debug\ndebug: true\n\n# Do not output anything except error\nquiet: false\n```\n\n#### Handled command matching\n\nAll settings can be mapped to an handled command.\n\nFor instance:\n\n```yaml\n\n# This config will handle command `abc` and `xyz`\ncommands:\n  - abc\n  - xyz\n\n# you can map commands to handled commands bellow\npre_conf_commands:\n  - abc:\n    - echo "will run for command abc"\n  - xyz:\n    - echo "will run for command xyz"\n    - echo "Can be multiple"\n  - echo "Will run for both commands"\n\nuser:\n  - abc: 1000\n  - xyz: 1001\n\n# Mapping can also be a dictionnary\ngroup:\n  abc: 1000\n  xyz: 1001\n\n# Etc\n```\n\nNot supported for deprecated settings `command`, `subcommands` and `links`.\n\n\n\n### Config templates\n\nYou can generate configuration for your service with jinja2 template.\n\n**`links` and `containers` are not supported with docker network and docker-compose v2.**\n\nHere is an example for an hypothetical ssh config file:\n\n```jinja\nhost server:\n    hostname {{links.ssh.ip}}\n    port {{links.ssh.port}}\n```\n\nTemplates will be replaced with ip address and port of the identified link. All links can be accessed from `links.all`, this is a tuple of links you can iterate on it.\n\n```jinja\n{% for link in links.all %}\nhost {{link.names[0]}}\n    hostname {{link.ip}}\n    port {{links.port}}\n{% endfor %}\n```\n\nIf you change the option `single` to `false` in the `entrypoint-config.yml`, the identified link `ssh` will become a tuple of links. You must iterate on it in the `jinja` template.\n\n```jinja\n{% for link in links.ssh %}\nhost {{link.names[0]}}\n    hostname {{link.ip}}\n    port {{links.port}}\n{% endfor %}\n```\n\nAccessing environment in template.\n\n```jinja\n{% if \'SSHKEY in env\' %}\n{{env[\'SSHKEY\']}}\n{% endfor %}\n```\n\n### Accessible objects\n\nYou have 4 available objects in your templates.\n\n - `config`\n - `links`\n - `containers`\n - `environ`\n - `yaml`\n - `json`\n\n#### config\n\n`Config` reflect the config file. You can retrieve any setup in this object.\n\n(see `config.py`)\n\n#### links\n\n**Not supported with docker network and docker-compose v2**\n\n`Links` handles `Link` objects. You can identify links using wildcard patterns in the configuration file.\n\n`link` is related to one physical link (one ip and one port).\n\n`link` handles the following attributes:\n  - `ip`\n    - link ip\n  - `port`\n    - link port (integer)\n  - `environ`\n    - related container environment\n  - `protocol`\n    - link protocol (`tcp` or `udp`)\n  - `uri`\n    - link URI (example: `tcp://10.0.0.3:80`)\n  - `names`\n    - tuple of related container names\n\n#### containers\n\n**Not supported with docker network and docker-compose v2**\n\n`containers` handles a tuple of `container` object.\n\n`container` handles the following attributes:\n  - `ip`\n    - container ip\n  - `environ`\n    - container environment\n  - `names`\n    - List of containers names\n      - Names are sorted by length, but container ID will be the last element.\n  - `id`\n    - Hexadecimal container ID (if available, empty string else)\n  - `links`\n    - Tuple of `link` objects related to this container\n\n#### environ\n`environ` is the environment of the container (os.environ).\n\n`env` is an alias to `environ`.\n\n##### envtobool\n`envtobool` function is a useful to parse boolean string input in environnement to enable or disable features.\n\nThe function accepts a default value as second parameter.\n\n```jinja\n{% if envtobool(\'SOME_ENV_VARIABLE\', False) %}\ndo stuff\n{% endif %}\n\n# Will write True or False here\n{envtobool(\'SOME_OTHER_ENV_VARIABLE\', True)}\n```\n\nSee https://docs.python.org/3/distutils/apiref.html#distutils.util.strtobool for information on input.\n\n#### yaml and json\n\n`yaml` and `json` objects are respectively an import of [`PyYAML`](http://pyyaml.org/) and [`json`](https://docs.python.org/2/library/json.html) modules.\n\nThey are useful to load and dump serialized data from environment.\n\n```jinja\n# Here yaml is present in SETUP_YAML environment variable\n{% set data = yaml.load(env[\'SETUP_YAML\'])%}\n{{data[\'param\']}}\n\n# Here json is present in SETUP_JSON environment variable\n{% set data = json.loads(env[\'SETUP_JSON\'])%}\n{{data[\'param\']}}\n```\n\n## Setup\n\nSome setups can be overridden using environment variables.\n\n- `ENTRYPOINT_CONFIG` overrides path of `entrypoint-config.yml` file.\n- `ENTRYPOINT_FORCE` is applying configuration and runs pre and post conf commands even if the `command` provided is not handled.\n- `ENTRYPOINT_PRECONF_COMMAND` run an extra pre conf shell command after all pre conf commands.\n- `ENTRYPOINT_POSTCONF_COMMAND` run an extra post conf shell command after all post conf commands.\n- `ENTRYPOINT_DEBUG` enables debug logs.\n- `ENTRYPOINT_RAW` does not use logging to display pre and post conf commands.\n   This can be useful if output is serialized.\n- `ENTRYPOINT_DISABLE_RELOAD` disable reload system even if it is enabled in `entrypoint-config.yml`.\n- `ENTRYPOINT_USER` overrides `user` in config.\n- `ENTRYPOINT_GROUP` overrides `group` in config.\n- `ENTRYPOINT_DISABLE_SERVICE` exits container with 0 before doing anything. Useful to disable container using environement.\n\n\n\n### Running Tests\n\nTo run tests, ensure that `docker-compose` and `make` are installed and run\n\n```shell\n$ make test\n```\n',
    'author': 'Christophe Mehay',
    'author_email': 'cmehay@nospam.student.42.fr',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'http://github.com/cmehay/pyentrypoint',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
