"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UnsupportedLanguageError = exports.DocumentationLanguage = void 0;
/**
 * Supported languages to generate documentation in.
 */
class DocumentationLanguage {
    constructor(lang) {
        this.lang = lang;
    }
    /**
     * Transform a literal string to the `DocumentationLanguage` object.
     *
     * Throws an `UnsupportedLanguageError` if the language is not supported.
     */
    static fromString(lang) {
        switch (lang) {
            case DocumentationLanguage.TYPESCRIPT.toString():
                return DocumentationLanguage.TYPESCRIPT;
            case DocumentationLanguage.PYTHON.toString():
                return DocumentationLanguage.PYTHON;
            case DocumentationLanguage.JAVA.toString():
                return DocumentationLanguage.JAVA;
            default:
                throw new UnsupportedLanguageError(lang, DocumentationLanguage.ALL);
        }
    }
    toString() {
        return this.lang;
    }
}
exports.DocumentationLanguage = DocumentationLanguage;
/**
 * TypeScript.
 */
DocumentationLanguage.TYPESCRIPT = new DocumentationLanguage('typescript');
/**
 * Python.
 */
DocumentationLanguage.PYTHON = new DocumentationLanguage('python');
/**
 * Java.
 */
DocumentationLanguage.JAVA = new DocumentationLanguage('java');
/**
 * All supported languages.
 */
DocumentationLanguage.ALL = [
    DocumentationLanguage.TYPESCRIPT,
    DocumentationLanguage.PYTHON,
    DocumentationLanguage.JAVA,
];
class UnsupportedLanguageError extends Error {
    constructor(lang, supported) {
        super(`Unsupported language: ${lang}. Supported languages are: [${supported}]`);
    }
}
exports.UnsupportedLanguageError = UnsupportedLanguageError;
//# sourceMappingURL=data:application/json;base64,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