"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LicenseList = void 0;
const fs = require("fs");
const os = require("os");
const path = require("path");
const s3 = require("@aws-cdk/aws-s3");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const core_1 = require("@aws-cdk/core");
/**
 * A list of licenses, which can be used to control filter packages before
 * indexing.
 */
class LicenseList extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The object key i which the license list is stored.
         */
        this.objectKey = 'allowed-licenses.json';
        this.bucket = new s3.Bucket(this, 'Bucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            versioned: true,
        });
        this.upload = new s3deploy.BucketDeployment(this, 'Resource', {
            destinationBucket: this.bucket,
            prune: true,
            retainOnDelete: false,
            sources: [this.createAsset(props.licenses)],
        });
    }
    /**
     * Grants an AWS Lambda function permissions to read the license allow list,
     * and adds the relevant environment variables expected by the
     * `LicenseListClient`.
     */
    grantRead(handler) {
        handler.addEnvironment("LICENSE_LIST_BUCKET_NAME" /* BUCKET_NAME */, this.bucket.bucketName);
        handler.addEnvironment("LICENSE_LIST_OBJECT_KEY" /* OBJECT_KEY */, this.objectKey);
        this.bucket.grantRead(handler);
        // The handler now depends on the deny-list having been uploaded
        handler.node.addDependency(this.upload);
    }
    createAsset(licenses) {
        const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'license-list-'));
        fs.writeFileSync(path.join(tmpdir, this.objectKey), JSON.stringify(licenses.map((l) => l.id), null, 2), 'utf-8');
        return s3deploy.Source.asset(tmpdir);
    }
}
exports.LicenseList = LicenseList;
//# sourceMappingURL=data:application/json;base64,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