"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const crypto_1 = require("crypto");
const path_1 = require("path");
const url_1 = require("url");
const spec_1 = require("@jsii/spec");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const Environments_1 = require("aws-embedded-metrics/lib/environment/Environments");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
const tarball_lambda_shared_1 = require("../shared/tarball.lambda-shared");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.environmentOverride = Environments_1.default.Lambda;
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, context) => {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // Clear out the default dimensions, we won't need them.
    metrics.setDimensions();
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const STATE_MACHINE_ARN = env_lambda_shared_1.requireEnv('STATE_MACHINE_ARN');
    const PACKAGE_LINKS = env_lambda_shared_1.requireEnv('PACKAGE_LINKS');
    const result = new Array();
    for (const record of (_a = event.Records) !== null && _a !== void 0 ? _a : []) {
        const payload = JSON.parse(record.body);
        const tarballUri = new url_1.URL(payload.tarballUri);
        if (tarballUri.protocol !== 's3:') {
            throw new Error(`Unsupported protocol in URI: ${tarballUri}`);
        }
        const tarball = await aws
            .s3()
            .getObject({
            // Note: we drop anything after the first `.` in the host, as we only care about the bucket name.
            Bucket: tarballUri.host.split('.')[0],
            // Note: the pathname part is absolute, so we strip the leading `/`.
            Key: tarballUri.pathname.replace(/^\//, ''),
            VersionId: (_b = tarballUri.searchParams.get('versionId')) !== null && _b !== void 0 ? _b : undefined,
        })
            .promise();
        const integrityCheck = integrity_lambda_shared_1.integrity(payload, Buffer.from(tarball.Body));
        if (payload.integrity !== integrityCheck) {
            throw new Error(`Integrity check failed: ${payload.integrity} !== ${integrityCheck}`);
        }
        let dotJsii;
        let packageJson;
        let licenseText;
        try {
            ({ dotJsii, packageJson, licenseText } = await tarball_lambda_shared_1.extractObjects(Buffer.from(tarball.Body), {
                dotJsii: { path: 'package/.jsii', required: true },
                packageJson: { path: 'package/package.json', required: true },
                licenseText: { filter: isLicenseFile },
            }));
        }
        catch (err) {
            console.error(`Invalid tarball content: ${err}`);
            metrics.putMetric("InvalidTarball" /* INVALID_TARBALL */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        let packageLicense;
        let packageName;
        let packageVersion;
        try {
            const { license, name, version } = spec_1.validateAssembly(JSON.parse(dotJsii.toString('utf-8')));
            packageLicense = license;
            packageName = name;
            packageVersion = version;
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (ex) {
            console.error(`Package does not contain a valid assembly -- ignoring: ${ex}`);
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        // Ensure the `.jsii` name, version & license corresponds to those in `package.json`
        const { name: packageJsonName, version: packageJsonVersion, license: packageJsonLicense, constructHub, } = JSON.parse(packageJson.toString('utf-8'));
        if (packageJsonName !== packageName ||
            packageJsonVersion !== packageVersion ||
            packageJsonLicense !== packageLicense) {
            console.log(`Ignoring package with mismatched name, version, and/or license (${packageJsonName}@${packageJsonVersion} is ${packageJsonLicense} !== ${packageName}@${packageVersion} is ${packageLicense})`);
            metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 0, aws_embedded_metrics_1.Unit.Count);
        // Did we identify a license file or not?
        metrics.putMetric("FoundLicenseFile" /* FOUND_LICENSE_FILE */, licenseText != null ? 1 : 0, aws_embedded_metrics_1.Unit.Count);
        // Add custom links content to metdata for display on the frontend
        const allowedLinks = JSON.parse(PACKAGE_LINKS);
        const packageLinks = allowedLinks.reduce((accum, { configKey, allowedDomains }) => {
            const pkgValue = constructHub === null || constructHub === void 0 ? void 0 : constructHub.packageLinks[configKey];
            if (!pkgValue) {
                return accum;
            }
            // check if value is in allowed domains list
            const url = new url_1.URL(pkgValue);
            if ((allowedDomains === null || allowedDomains === void 0 ? void 0 : allowedDomains.length) && !allowedDomains.includes(url.host)) {
                return accum;
            }
            // if no allow list is provided
            return { ...accum, [configKey]: pkgValue };
        }, {});
        const metadata = {
            date: payload.time,
            licenseText: licenseText === null || licenseText === void 0 ? void 0 : licenseText.toString('utf-8'),
            packageLinks,
        };
        const { assemblyKey, metadataKey, packageKey } = constants.getObjectKeys(packageName, packageVersion);
        console.log(`Writing assembly at ${assemblyKey}`);
        console.log(`Writing package at  ${packageKey}`);
        console.log(`Writing metadata at  ${metadataKey}`);
        // we upload the metadata file first because the catalog builder depends on
        // it and is triggered by the assembly file upload.
        console.log(`${packageName}@${packageVersion} | Uploading package and metadata files`);
        const [pkg, storedMetadata] = await Promise.all([
            aws
                .s3()
                .putObject({
                Bucket: BUCKET_NAME,
                Key: packageKey,
                Body: tarball.Body,
                CacheControl: 'public',
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            })
                .promise(),
            aws
                .s3()
                .putObject({
                Bucket: BUCKET_NAME,
                Key: metadataKey,
                Body: JSON.stringify(metadata),
                CacheControl: 'public',
                ContentType: 'application/json',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            })
                .promise(),
        ]);
        // now we can upload the assembly.
        console.log(`${packageName}@${packageVersion} | Uploading assembly file`);
        const assembly = await aws
            .s3()
            .putObject({
            Bucket: BUCKET_NAME,
            Key: assemblyKey,
            Body: dotJsii,
            CacheControl: 'public',
            ContentType: 'application/json',
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
            },
        })
            .promise();
        const created = {
            bucket: BUCKET_NAME,
            assembly: {
                key: assemblyKey,
                versionId: assembly.VersionId,
            },
            package: {
                key: packageKey,
                versionId: pkg.VersionId,
            },
            metadata: {
                key: metadataKey,
                versionId: storedMetadata.VersionId,
            },
        };
        console.log(`Created objects: ${JSON.stringify(created, null, 2)}`);
        const sfn = await aws
            .stepFunctions()
            .startExecution({
            input: JSON.stringify(created),
            name: sfnExecutionNameFromParts(packageName, `v${packageVersion}`, context.awsRequestId),
            stateMachineArn: STATE_MACHINE_ARN,
        })
            .promise();
        console.log(`Started StateMachine execution: ${sfn.executionArn}`);
        result.push(sfn.executionArn);
    }
    return result;
});
/**
 * Checks whether the provided file name corresponds to a license file or not.
 *
 * @param fileName the file name to be checked.
 *
 * @returns `true` IIF the file is named LICENSE and has the .MD or .TXT
 *          extension, or no extension at all. The test is case-insensitive.
 */
function isLicenseFile(fileName) {
    const ext = path_1.extname(fileName);
    const possibleExtensions = new Set(['', '.md', '.txt']);
    return (possibleExtensions.has(ext.toLowerCase()) &&
        path_1.basename(fileName, ext).toUpperCase() === 'LICENSE');
}
/**
 * Creates a StepFunction execution request name based on the provided parts.
 * The result is guaranteed to be 80 characters or less and to contain only
 * characters that are valid for a StepFunction execution request name for which
 * CloudWatch Logging can be enabled. The resulting name is very likely to
 * be unique for a given input.
 */
function sfnExecutionNameFromParts(first, ...rest) {
    const parts = [first, ...rest];
    const name = parts
        .map((part) => part.replace(/[^a-z0-9_-]+/gi, '_'))
        .join('_')
        .replace(/^_/g, '')
        .replace(/_{2,}/g, '_');
    if (name.length <= 80) {
        return name;
    }
    const suffix = crypto_1.createHash('sha256')
        // The hash is computed based on input arguments, to maximize unicity
        .update(parts.join('_'))
        .digest('hex')
        .substring(0, 6);
    return `${name.substring(0, 80 - suffix.length - 1)}_${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5nZXN0aW9uLmxhbWJkYS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9iYWNrZW5kL2luZ2VzdGlvbi9pbmdlc3Rpb24ubGFtYmRhLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLG1DQUFvQztBQUNwQywrQkFBeUM7QUFDekMsNkJBQTBCO0FBRTFCLHFDQUE4QztBQUM5QywrREFBd0U7QUFDeEUsb0ZBQTZFO0FBSTdFLG1EQUFtRDtBQUNuRCxpREFBaUQ7QUFDakQsbUVBQXlEO0FBRXpELCtFQUE4RDtBQUM5RCwyRUFBaUU7QUFDakUsMkNBQTREO0FBRTVELG9DQUFhLENBQUMsbUJBQW1CLEdBQUcsc0JBQVksQ0FBQyxNQUFNLENBQUM7QUFDeEQsb0NBQWEsQ0FBQyxTQUFTLEdBQUcsNkJBQWlCLENBQUM7QUFFL0IsUUFBQSxPQUFPLEdBQUcsa0NBQVcsQ0FDaEMsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLEtBQUssRUFBRSxLQUFlLEVBQUUsT0FBZ0IsRUFBRSxFQUFFOztJQUN2RCxPQUFPLENBQUMsR0FBRyxDQUFDLFVBQVUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUV4RCx3REFBd0Q7SUFDeEQsT0FBTyxDQUFDLGFBQWEsRUFBRSxDQUFDO0lBRXhCLE1BQU0sV0FBVyxHQUFHLDhCQUFVLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDOUMsTUFBTSxpQkFBaUIsR0FBRyw4QkFBVSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDMUQsTUFBTSxhQUFhLEdBQUcsOEJBQVUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUVsRCxNQUFNLE1BQU0sR0FBRyxJQUFJLEtBQUssRUFBVSxDQUFDO0lBRW5DLEtBQUssTUFBTSxNQUFNLFVBQUksS0FBSyxDQUFDLE9BQU8sbUNBQUksRUFBRSxFQUFFO1FBQ3hDLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBbUIsQ0FBQztRQUUxRCxNQUFNLFVBQVUsR0FBRyxJQUFJLFNBQUcsQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDL0MsSUFBSSxVQUFVLENBQUMsUUFBUSxLQUFLLEtBQUssRUFBRTtZQUNqQyxNQUFNLElBQUksS0FBSyxDQUFDLGdDQUFnQyxVQUFVLEVBQUUsQ0FBQyxDQUFDO1NBQy9EO1FBQ0QsTUFBTSxPQUFPLEdBQUcsTUFBTSxHQUFHO2FBQ3RCLEVBQUUsRUFBRTthQUNKLFNBQVMsQ0FBQztZQUNULGlHQUFpRztZQUNqRyxNQUFNLEVBQUUsVUFBVSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ3JDLG9FQUFvRTtZQUNwRSxHQUFHLEVBQUUsVUFBVSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQztZQUMzQyxTQUFTLFFBQUUsVUFBVSxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFDLG1DQUFJLFNBQVM7U0FDakUsQ0FBQzthQUNELE9BQU8sRUFBRSxDQUFDO1FBRWIsTUFBTSxjQUFjLEdBQUcsbUNBQVMsQ0FBQyxPQUFPLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSyxDQUFDLENBQUMsQ0FBQztRQUN0RSxJQUFJLE9BQU8sQ0FBQyxTQUFTLEtBQUssY0FBYyxFQUFFO1lBQ3hDLE1BQU0sSUFBSSxLQUFLLENBQ2IsMkJBQTJCLE9BQU8sQ0FBQyxTQUFTLFFBQVEsY0FBYyxFQUFFLENBQ3JFLENBQUM7U0FDSDtRQUVELElBQUksT0FBZSxDQUFDO1FBQ3BCLElBQUksV0FBbUIsQ0FBQztRQUN4QixJQUFJLFdBQStCLENBQUM7UUFDcEMsSUFBSTtZQUNGLENBQUMsRUFBRSxPQUFPLEVBQUUsV0FBVyxFQUFFLFdBQVcsRUFBRSxHQUFHLE1BQU0sc0NBQWMsQ0FDM0QsTUFBTSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSyxDQUFDLEVBQzFCO2dCQUNFLE9BQU8sRUFBRSxFQUFFLElBQUksRUFBRSxlQUFlLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRTtnQkFDbEQsV0FBVyxFQUFFLEVBQUUsSUFBSSxFQUFFLHNCQUFzQixFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUU7Z0JBQzdELFdBQVcsRUFBRSxFQUFFLE1BQU0sRUFBRSxhQUFhLEVBQUU7YUFDdkMsQ0FDRixDQUFDLENBQUM7U0FDSjtRQUFDLE9BQU8sR0FBRyxFQUFFO1lBQ1osT0FBTyxDQUFDLEtBQUssQ0FBQyw0QkFBNEIsR0FBRyxFQUFFLENBQUMsQ0FBQztZQUNqRCxPQUFPLENBQUMsU0FBUyx5Q0FBNkIsQ0FBQyxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDN0QsT0FBTztTQUNSO1FBRUQsSUFBSSxjQUFzQixDQUFDO1FBQzNCLElBQUksV0FBbUIsQ0FBQztRQUN4QixJQUFJLGNBQXNCLENBQUM7UUFDM0IsSUFBSTtZQUNGLE1BQU0sRUFBRSxPQUFPLEVBQUUsSUFBSSxFQUFFLE9BQU8sRUFBRSxHQUFHLHVCQUFnQixDQUNqRCxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FDdEMsQ0FBQztZQUNGLGNBQWMsR0FBRyxPQUFPLENBQUM7WUFDekIsV0FBVyxHQUFHLElBQUksQ0FBQztZQUNuQixjQUFjLEdBQUcsT0FBTyxDQUFDO1lBQ3pCLE9BQU8sQ0FBQyxTQUFTLDJDQUE4QixDQUFDLEVBQUUsMkJBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztTQUMvRDtRQUFDLE9BQU8sRUFBRSxFQUFFO1lBQ1gsT0FBTyxDQUFDLEtBQUssQ0FDWCwwREFBMEQsRUFBRSxFQUFFLENBQy9ELENBQUM7WUFDRixPQUFPLENBQUMsU0FBUywyQ0FBOEIsQ0FBQyxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDOUQsT0FBTztTQUNSO1FBRUQsb0ZBQW9GO1FBQ3BGLE1BQU0sRUFDSixJQUFJLEVBQUUsZUFBZSxFQUNyQixPQUFPLEVBQUUsa0JBQWtCLEVBQzNCLE9BQU8sRUFBRSxrQkFBa0IsRUFDM0IsWUFBWSxHQUNiLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7UUFDOUMsSUFDRSxlQUFlLEtBQUssV0FBVztZQUMvQixrQkFBa0IsS0FBSyxjQUFjO1lBQ3JDLGtCQUFrQixLQUFLLGNBQWMsRUFDckM7WUFDQSxPQUFPLENBQUMsR0FBRyxDQUNULG1FQUFtRSxlQUFlLElBQUksa0JBQWtCLE9BQU8sa0JBQWtCLFFBQVEsV0FBVyxJQUFJLGNBQWMsT0FBTyxjQUFjLEdBQUcsQ0FDL0wsQ0FBQztZQUNGLE9BQU8sQ0FBQyxTQUFTLHNFQUVmLENBQUMsRUFDRCwyQkFBSSxDQUFDLEtBQUssQ0FDWCxDQUFDO1lBQ0YsU0FBUztTQUNWO1FBQ0QsT0FBTyxDQUFDLFNBQVMsc0VBRWYsQ0FBQyxFQUNELDJCQUFJLENBQUMsS0FBSyxDQUNYLENBQUM7UUFFRix5Q0FBeUM7UUFDekMsT0FBTyxDQUFDLFNBQVMsOENBRWYsV0FBVyxJQUFJLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQzNCLDJCQUFJLENBQUMsS0FBSyxDQUNYLENBQUM7UUFFRixrRUFBa0U7UUFDbEUsTUFBTSxZQUFZLEdBQXdCLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUM7UUFFcEUsTUFBTSxZQUFZLEdBQUcsWUFBWSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEtBQUssRUFBRSxFQUFFLFNBQVMsRUFBRSxjQUFjLEVBQUUsRUFBRSxFQUFFO1lBQ2hGLE1BQU0sUUFBUSxHQUFHLFlBQVksYUFBWixZQUFZLHVCQUFaLFlBQVksQ0FBRSxZQUFZLENBQUMsU0FBUyxDQUFDLENBQUM7WUFFdkQsSUFBSSxDQUFDLFFBQVEsRUFBRTtnQkFDYixPQUFPLEtBQUssQ0FBQzthQUNkO1lBRUQsNENBQTRDO1lBQzVDLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQzlCLElBQUksQ0FBQSxjQUFjLGFBQWQsY0FBYyx1QkFBZCxjQUFjLENBQUUsTUFBTSxLQUFJLENBQUMsY0FBYyxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUU7Z0JBQ2hFLE9BQU8sS0FBSyxDQUFDO2FBQ2Q7WUFFRCwrQkFBK0I7WUFDL0IsT0FBTyxFQUFFLEdBQUcsS0FBSyxFQUFFLENBQUMsU0FBUyxDQUFDLEVBQUUsUUFBUSxFQUFFLENBQUM7UUFDN0MsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRVAsTUFBTSxRQUFRLEdBQUc7WUFDZixJQUFJLEVBQUUsT0FBTyxDQUFDLElBQUk7WUFDbEIsV0FBVyxFQUFFLFdBQVcsYUFBWCxXQUFXLHVCQUFYLFdBQVcsQ0FBRSxRQUFRLENBQUMsT0FBTyxDQUFDO1lBQzNDLFlBQVk7U0FDYixDQUFDO1FBRUYsTUFBTSxFQUFFLFdBQVcsRUFBRSxXQUFXLEVBQUUsVUFBVSxFQUFFLEdBQUcsU0FBUyxDQUFDLGFBQWEsQ0FDdEUsV0FBVyxFQUNYLGNBQWMsQ0FDZixDQUFDO1FBQ0YsT0FBTyxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsV0FBVyxFQUFFLENBQUMsQ0FBQztRQUNsRCxPQUFPLENBQUMsR0FBRyxDQUFDLHVCQUF1QixVQUFVLEVBQUUsQ0FBQyxDQUFDO1FBQ2pELE9BQU8sQ0FBQyxHQUFHLENBQUMsd0JBQXdCLFdBQVcsRUFBRSxDQUFDLENBQUM7UUFFbkQsMkVBQTJFO1FBQzNFLG1EQUFtRDtRQUNuRCxPQUFPLENBQUMsR0FBRyxDQUNULEdBQUcsV0FBVyxJQUFJLGNBQWMseUNBQXlDLENBQzFFLENBQUM7UUFDRixNQUFNLENBQUMsR0FBRyxFQUFFLGNBQWMsQ0FBQyxHQUFHLE1BQU0sT0FBTyxDQUFDLEdBQUcsQ0FBQztZQUM5QyxHQUFHO2lCQUNBLEVBQUUsRUFBRTtpQkFDSixTQUFTLENBQUM7Z0JBQ1QsTUFBTSxFQUFFLFdBQVc7Z0JBQ25CLEdBQUcsRUFBRSxVQUFVO2dCQUNmLElBQUksRUFBRSxPQUFPLENBQUMsSUFBSTtnQkFDbEIsWUFBWSxFQUFFLFFBQVE7Z0JBQ3RCLFdBQVcsRUFBRSwwQkFBMEI7Z0JBQ3ZDLFFBQVEsRUFBRTtvQkFDUixrQkFBa0IsRUFBRSxPQUFPLENBQUMsWUFBWTtvQkFDeEMsbUJBQW1CLEVBQUUsT0FBTyxDQUFDLGFBQWE7b0JBQzFDLGVBQWUsRUFBRSxPQUFPLENBQUMsWUFBWTtpQkFDdEM7YUFDRixDQUFDO2lCQUNELE9BQU8sRUFBRTtZQUNaLEdBQUc7aUJBQ0EsRUFBRSxFQUFFO2lCQUNKLFNBQVMsQ0FBQztnQkFDVCxNQUFNLEVBQUUsV0FBVztnQkFDbkIsR0FBRyxFQUFFLFdBQVc7Z0JBQ2hCLElBQUksRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQztnQkFDOUIsWUFBWSxFQUFFLFFBQVE7Z0JBQ3RCLFdBQVcsRUFBRSxrQkFBa0I7Z0JBQy9CLFFBQVEsRUFBRTtvQkFDUixrQkFBa0IsRUFBRSxPQUFPLENBQUMsWUFBWTtvQkFDeEMsbUJBQW1CLEVBQUUsT0FBTyxDQUFDLGFBQWE7b0JBQzFDLGVBQWUsRUFBRSxPQUFPLENBQUMsWUFBWTtpQkFDdEM7YUFDRixDQUFDO2lCQUNELE9BQU8sRUFBRTtTQUNiLENBQUMsQ0FBQztRQUVILGtDQUFrQztRQUNsQyxPQUFPLENBQUMsR0FBRyxDQUFDLEdBQUcsV0FBVyxJQUFJLGNBQWMsNEJBQTRCLENBQUMsQ0FBQztRQUMxRSxNQUFNLFFBQVEsR0FBRyxNQUFNLEdBQUc7YUFDdkIsRUFBRSxFQUFFO2FBQ0osU0FBUyxDQUFDO1lBQ1QsTUFBTSxFQUFFLFdBQVc7WUFDbkIsR0FBRyxFQUFFLFdBQVc7WUFDaEIsSUFBSSxFQUFFLE9BQU87WUFDYixZQUFZLEVBQUUsUUFBUTtZQUN0QixXQUFXLEVBQUUsa0JBQWtCO1lBQy9CLFFBQVEsRUFBRTtnQkFDUixrQkFBa0IsRUFBRSxPQUFPLENBQUMsWUFBWTtnQkFDeEMsbUJBQW1CLEVBQUUsT0FBTyxDQUFDLGFBQWE7Z0JBQzFDLGVBQWUsRUFBRSxPQUFPLENBQUMsWUFBWTthQUN0QztTQUNGLENBQUM7YUFDRCxPQUFPLEVBQUUsQ0FBQztRQUViLE1BQU0sT0FBTyxHQUFzQjtZQUNqQyxNQUFNLEVBQUUsV0FBVztZQUNuQixRQUFRLEVBQUU7Z0JBQ1IsR0FBRyxFQUFFLFdBQVc7Z0JBQ2hCLFNBQVMsRUFBRSxRQUFRLENBQUMsU0FBUzthQUM5QjtZQUNELE9BQU8sRUFBRTtnQkFDUCxHQUFHLEVBQUUsVUFBVTtnQkFDZixTQUFTLEVBQUUsR0FBRyxDQUFDLFNBQVM7YUFDekI7WUFDRCxRQUFRLEVBQUU7Z0JBQ1IsR0FBRyxFQUFFLFdBQVc7Z0JBQ2hCLFNBQVMsRUFBRSxjQUFjLENBQUMsU0FBUzthQUNwQztTQUNGLENBQUM7UUFDRixPQUFPLENBQUMsR0FBRyxDQUFDLG9CQUFvQixJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBRXBFLE1BQU0sR0FBRyxHQUFHLE1BQU0sR0FBRzthQUNsQixhQUFhLEVBQUU7YUFDZixjQUFjLENBQUM7WUFDZCxLQUFLLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUM7WUFDOUIsSUFBSSxFQUFFLHlCQUF5QixDQUM3QixXQUFXLEVBQ1gsSUFBSSxjQUFjLEVBQUUsRUFDcEIsT0FBTyxDQUFDLFlBQVksQ0FDckI7WUFDRCxlQUFlLEVBQUUsaUJBQWlCO1NBQ25DLENBQUM7YUFDRCxPQUFPLEVBQUUsQ0FBQztRQUNiLE9BQU8sQ0FBQyxHQUFHLENBQUMsbUNBQW1DLEdBQUcsQ0FBQyxZQUFZLEVBQUUsQ0FBQyxDQUFDO1FBQ25FLE1BQU0sQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxDQUFDO0tBQy9CO0lBRUQsT0FBTyxNQUFNLENBQUM7QUFDaEIsQ0FBQyxDQUNGLENBQUM7QUFFRjs7Ozs7OztHQU9HO0FBQ0gsU0FBUyxhQUFhLENBQUMsUUFBZ0I7SUFDckMsTUFBTSxHQUFHLEdBQUcsY0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQzlCLE1BQU0sa0JBQWtCLEdBQUcsSUFBSSxHQUFHLENBQUMsQ0FBQyxFQUFFLEVBQUUsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDLENBQUM7SUFDeEQsT0FBTyxDQUNMLGtCQUFrQixDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsV0FBVyxFQUFFLENBQUM7UUFDekMsZUFBUSxDQUFDLFFBQVEsRUFBRSxHQUFHLENBQUMsQ0FBQyxXQUFXLEVBQUUsS0FBSyxTQUFTLENBQ3BELENBQUM7QUFDSixDQUFDO0FBRUQ7Ozs7OztHQU1HO0FBQ0gsU0FBUyx5QkFBeUIsQ0FDaEMsS0FBYSxFQUNiLEdBQUcsSUFBdUI7SUFFMUIsTUFBTSxLQUFLLEdBQUcsQ0FBQyxLQUFLLEVBQUUsR0FBRyxJQUFJLENBQUMsQ0FBQztJQUMvQixNQUFNLElBQUksR0FBRyxLQUFLO1NBQ2YsR0FBRyxDQUFDLENBQUMsSUFBSSxFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLGdCQUFnQixFQUFFLEdBQUcsQ0FBQyxDQUFDO1NBQ2xELElBQUksQ0FBQyxHQUFHLENBQUM7U0FDVCxPQUFPLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQztTQUNsQixPQUFPLENBQUMsUUFBUSxFQUFFLEdBQUcsQ0FBQyxDQUFDO0lBQzFCLElBQUksSUFBSSxDQUFDLE1BQU0sSUFBSSxFQUFFLEVBQUU7UUFDckIsT0FBTyxJQUFJLENBQUM7S0FDYjtJQUNELE1BQU0sTUFBTSxHQUFHLG1CQUFVLENBQUMsUUFBUSxDQUFDO1FBQ2pDLHFFQUFxRTtTQUNwRSxNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUN2QixNQUFNLENBQUMsS0FBSyxDQUFDO1NBQ2IsU0FBUyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUNuQixPQUFPLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxHQUFHLE1BQU0sQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLElBQUksTUFBTSxFQUFFLENBQUM7QUFDbEUsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGNyZWF0ZUhhc2ggfSBmcm9tICdjcnlwdG8nO1xuaW1wb3J0IHsgYmFzZW5hbWUsIGV4dG5hbWUgfSBmcm9tICdwYXRoJztcbmltcG9ydCB7IFVSTCB9IGZyb20gJ3VybCc7XG5cbmltcG9ydCB7IHZhbGlkYXRlQXNzZW1ibHkgfSBmcm9tICdAanNpaS9zcGVjJztcbmltcG9ydCB7IG1ldHJpY1Njb3BlLCBDb25maWd1cmF0aW9uLCBVbml0IH0gZnJvbSAnYXdzLWVtYmVkZGVkLW1ldHJpY3MnO1xuaW1wb3J0IEVudmlyb25tZW50cyBmcm9tICdhd3MtZW1iZWRkZWQtbWV0cmljcy9saWIvZW52aXJvbm1lbnQvRW52aXJvbm1lbnRzJztcbmltcG9ydCB0eXBlIHsgQ29udGV4dCwgU1FTRXZlbnQgfSBmcm9tICdhd3MtbGFtYmRhJztcbmltcG9ydCB0eXBlIHsgUGFja2FnZUxpbmtDb25maWcgfSBmcm9tICcuLi8uLi93ZWJhcHAnO1xuaW1wb3J0IHR5cGUgeyBTdGF0ZU1hY2hpbmVJbnB1dCB9IGZyb20gJy4uL3BheWxvYWQtc2NoZW1hJztcbmltcG9ydCAqIGFzIGF3cyBmcm9tICcuLi9zaGFyZWQvYXdzLmxhbWJkYS1zaGFyZWQnO1xuaW1wb3J0ICogYXMgY29uc3RhbnRzIGZyb20gJy4uL3NoYXJlZC9jb25zdGFudHMnO1xuaW1wb3J0IHsgcmVxdWlyZUVudiB9IGZyb20gJy4uL3NoYXJlZC9lbnYubGFtYmRhLXNoYXJlZCc7XG5pbXBvcnQgeyBJbmdlc3Rpb25JbnB1dCB9IGZyb20gJy4uL3NoYXJlZC9pbmdlc3Rpb24taW5wdXQubGFtYmRhLXNoYXJlZCc7XG5pbXBvcnQgeyBpbnRlZ3JpdHkgfSBmcm9tICcuLi9zaGFyZWQvaW50ZWdyaXR5LmxhbWJkYS1zaGFyZWQnO1xuaW1wb3J0IHsgZXh0cmFjdE9iamVjdHMgfSBmcm9tICcuLi9zaGFyZWQvdGFyYmFsbC5sYW1iZGEtc2hhcmVkJztcbmltcG9ydCB7IE1ldHJpY05hbWUsIE1FVFJJQ1NfTkFNRVNQQUNFIH0gZnJvbSAnLi9jb25zdGFudHMnO1xuXG5Db25maWd1cmF0aW9uLmVudmlyb25tZW50T3ZlcnJpZGUgPSBFbnZpcm9ubWVudHMuTGFtYmRhO1xuQ29uZmlndXJhdGlvbi5uYW1lc3BhY2UgPSBNRVRSSUNTX05BTUVTUEFDRTtcblxuZXhwb3J0IGNvbnN0IGhhbmRsZXIgPSBtZXRyaWNTY29wZShcbiAgKG1ldHJpY3MpID0+IGFzeW5jIChldmVudDogU1FTRXZlbnQsIGNvbnRleHQ6IENvbnRleHQpID0+IHtcbiAgICBjb25zb2xlLmxvZyhgRXZlbnQ6ICR7SlNPTi5zdHJpbmdpZnkoZXZlbnQsIG51bGwsIDIpfWApO1xuXG4gICAgLy8gQ2xlYXIgb3V0IHRoZSBkZWZhdWx0IGRpbWVuc2lvbnMsIHdlIHdvbid0IG5lZWQgdGhlbS5cbiAgICBtZXRyaWNzLnNldERpbWVuc2lvbnMoKTtcblxuICAgIGNvbnN0IEJVQ0tFVF9OQU1FID0gcmVxdWlyZUVudignQlVDS0VUX05BTUUnKTtcbiAgICBjb25zdCBTVEFURV9NQUNISU5FX0FSTiA9IHJlcXVpcmVFbnYoJ1NUQVRFX01BQ0hJTkVfQVJOJyk7XG4gICAgY29uc3QgUEFDS0FHRV9MSU5LUyA9IHJlcXVpcmVFbnYoJ1BBQ0tBR0VfTElOS1MnKTtcblxuICAgIGNvbnN0IHJlc3VsdCA9IG5ldyBBcnJheTxzdHJpbmc+KCk7XG5cbiAgICBmb3IgKGNvbnN0IHJlY29yZCBvZiBldmVudC5SZWNvcmRzID8/IFtdKSB7XG4gICAgICBjb25zdCBwYXlsb2FkID0gSlNPTi5wYXJzZShyZWNvcmQuYm9keSkgYXMgSW5nZXN0aW9uSW5wdXQ7XG5cbiAgICAgIGNvbnN0IHRhcmJhbGxVcmkgPSBuZXcgVVJMKHBheWxvYWQudGFyYmFsbFVyaSk7XG4gICAgICBpZiAodGFyYmFsbFVyaS5wcm90b2NvbCAhPT0gJ3MzOicpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbnN1cHBvcnRlZCBwcm90b2NvbCBpbiBVUkk6ICR7dGFyYmFsbFVyaX1gKTtcbiAgICAgIH1cbiAgICAgIGNvbnN0IHRhcmJhbGwgPSBhd2FpdCBhd3NcbiAgICAgICAgLnMzKClcbiAgICAgICAgLmdldE9iamVjdCh7XG4gICAgICAgICAgLy8gTm90ZTogd2UgZHJvcCBhbnl0aGluZyBhZnRlciB0aGUgZmlyc3QgYC5gIGluIHRoZSBob3N0LCBhcyB3ZSBvbmx5IGNhcmUgYWJvdXQgdGhlIGJ1Y2tldCBuYW1lLlxuICAgICAgICAgIEJ1Y2tldDogdGFyYmFsbFVyaS5ob3N0LnNwbGl0KCcuJylbMF0sXG4gICAgICAgICAgLy8gTm90ZTogdGhlIHBhdGhuYW1lIHBhcnQgaXMgYWJzb2x1dGUsIHNvIHdlIHN0cmlwIHRoZSBsZWFkaW5nIGAvYC5cbiAgICAgICAgICBLZXk6IHRhcmJhbGxVcmkucGF0aG5hbWUucmVwbGFjZSgvXlxcLy8sICcnKSxcbiAgICAgICAgICBWZXJzaW9uSWQ6IHRhcmJhbGxVcmkuc2VhcmNoUGFyYW1zLmdldCgndmVyc2lvbklkJykgPz8gdW5kZWZpbmVkLFxuICAgICAgICB9KVxuICAgICAgICAucHJvbWlzZSgpO1xuXG4gICAgICBjb25zdCBpbnRlZ3JpdHlDaGVjayA9IGludGVncml0eShwYXlsb2FkLCBCdWZmZXIuZnJvbSh0YXJiYWxsLkJvZHkhKSk7XG4gICAgICBpZiAocGF5bG9hZC5pbnRlZ3JpdHkgIT09IGludGVncml0eUNoZWNrKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICBgSW50ZWdyaXR5IGNoZWNrIGZhaWxlZDogJHtwYXlsb2FkLmludGVncml0eX0gIT09ICR7aW50ZWdyaXR5Q2hlY2t9YCxcbiAgICAgICAgKTtcbiAgICAgIH1cblxuICAgICAgbGV0IGRvdEpzaWk6IEJ1ZmZlcjtcbiAgICAgIGxldCBwYWNrYWdlSnNvbjogQnVmZmVyO1xuICAgICAgbGV0IGxpY2Vuc2VUZXh0OiBCdWZmZXIgfCB1bmRlZmluZWQ7XG4gICAgICB0cnkge1xuICAgICAgICAoeyBkb3RKc2lpLCBwYWNrYWdlSnNvbiwgbGljZW5zZVRleHQgfSA9IGF3YWl0IGV4dHJhY3RPYmplY3RzKFxuICAgICAgICAgIEJ1ZmZlci5mcm9tKHRhcmJhbGwuQm9keSEpLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIGRvdEpzaWk6IHsgcGF0aDogJ3BhY2thZ2UvLmpzaWknLCByZXF1aXJlZDogdHJ1ZSB9LFxuICAgICAgICAgICAgcGFja2FnZUpzb246IHsgcGF0aDogJ3BhY2thZ2UvcGFja2FnZS5qc29uJywgcmVxdWlyZWQ6IHRydWUgfSxcbiAgICAgICAgICAgIGxpY2Vuc2VUZXh0OiB7IGZpbHRlcjogaXNMaWNlbnNlRmlsZSB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICkpO1xuICAgICAgfSBjYXRjaCAoZXJyKSB7XG4gICAgICAgIGNvbnNvbGUuZXJyb3IoYEludmFsaWQgdGFyYmFsbCBjb250ZW50OiAke2Vycn1gKTtcbiAgICAgICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5JTlZBTElEX1RBUkJBTEwsIDEsIFVuaXQuQ291bnQpO1xuICAgICAgICByZXR1cm47XG4gICAgICB9XG5cbiAgICAgIGxldCBwYWNrYWdlTGljZW5zZTogc3RyaW5nO1xuICAgICAgbGV0IHBhY2thZ2VOYW1lOiBzdHJpbmc7XG4gICAgICBsZXQgcGFja2FnZVZlcnNpb246IHN0cmluZztcbiAgICAgIHRyeSB7XG4gICAgICAgIGNvbnN0IHsgbGljZW5zZSwgbmFtZSwgdmVyc2lvbiB9ID0gdmFsaWRhdGVBc3NlbWJseShcbiAgICAgICAgICBKU09OLnBhcnNlKGRvdEpzaWkudG9TdHJpbmcoJ3V0Zi04JykpLFxuICAgICAgICApO1xuICAgICAgICBwYWNrYWdlTGljZW5zZSA9IGxpY2Vuc2U7XG4gICAgICAgIHBhY2thZ2VOYW1lID0gbmFtZTtcbiAgICAgICAgcGFja2FnZVZlcnNpb24gPSB2ZXJzaW9uO1xuICAgICAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLklOVkFMSURfQVNTRU1CTFksIDAsIFVuaXQuQ291bnQpO1xuICAgICAgfSBjYXRjaCAoZXgpIHtcbiAgICAgICAgY29uc29sZS5lcnJvcihcbiAgICAgICAgICBgUGFja2FnZSBkb2VzIG5vdCBjb250YWluIGEgdmFsaWQgYXNzZW1ibHkgLS0gaWdub3Jpbmc6ICR7ZXh9YCxcbiAgICAgICAgKTtcbiAgICAgICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5JTlZBTElEX0FTU0VNQkxZLCAxLCBVbml0LkNvdW50KTtcbiAgICAgICAgcmV0dXJuO1xuICAgICAgfVxuXG4gICAgICAvLyBFbnN1cmUgdGhlIGAuanNpaWAgbmFtZSwgdmVyc2lvbiAmIGxpY2Vuc2UgY29ycmVzcG9uZHMgdG8gdGhvc2UgaW4gYHBhY2thZ2UuanNvbmBcbiAgICAgIGNvbnN0IHtcbiAgICAgICAgbmFtZTogcGFja2FnZUpzb25OYW1lLFxuICAgICAgICB2ZXJzaW9uOiBwYWNrYWdlSnNvblZlcnNpb24sXG4gICAgICAgIGxpY2Vuc2U6IHBhY2thZ2VKc29uTGljZW5zZSxcbiAgICAgICAgY29uc3RydWN0SHViLFxuICAgICAgfSA9IEpTT04ucGFyc2UocGFja2FnZUpzb24udG9TdHJpbmcoJ3V0Zi04JykpO1xuICAgICAgaWYgKFxuICAgICAgICBwYWNrYWdlSnNvbk5hbWUgIT09IHBhY2thZ2VOYW1lIHx8XG4gICAgICAgIHBhY2thZ2VKc29uVmVyc2lvbiAhPT0gcGFja2FnZVZlcnNpb24gfHxcbiAgICAgICAgcGFja2FnZUpzb25MaWNlbnNlICE9PSBwYWNrYWdlTGljZW5zZVxuICAgICAgKSB7XG4gICAgICAgIGNvbnNvbGUubG9nKFxuICAgICAgICAgIGBJZ25vcmluZyBwYWNrYWdlIHdpdGggbWlzbWF0Y2hlZCBuYW1lLCB2ZXJzaW9uLCBhbmQvb3IgbGljZW5zZSAoJHtwYWNrYWdlSnNvbk5hbWV9QCR7cGFja2FnZUpzb25WZXJzaW9ufSBpcyAke3BhY2thZ2VKc29uTGljZW5zZX0gIT09ICR7cGFja2FnZU5hbWV9QCR7cGFja2FnZVZlcnNpb259IGlzICR7cGFja2FnZUxpY2Vuc2V9KWAsXG4gICAgICAgICk7XG4gICAgICAgIG1ldHJpY3MucHV0TWV0cmljKFxuICAgICAgICAgIE1ldHJpY05hbWUuTUlTTUFUQ0hFRF9JREVOVElUWV9SRUpFQ1RJT05TLFxuICAgICAgICAgIDEsXG4gICAgICAgICAgVW5pdC5Db3VudCxcbiAgICAgICAgKTtcbiAgICAgICAgY29udGludWU7XG4gICAgICB9XG4gICAgICBtZXRyaWNzLnB1dE1ldHJpYyhcbiAgICAgICAgTWV0cmljTmFtZS5NSVNNQVRDSEVEX0lERU5USVRZX1JFSkVDVElPTlMsXG4gICAgICAgIDAsXG4gICAgICAgIFVuaXQuQ291bnQsXG4gICAgICApO1xuXG4gICAgICAvLyBEaWQgd2UgaWRlbnRpZnkgYSBsaWNlbnNlIGZpbGUgb3Igbm90P1xuICAgICAgbWV0cmljcy5wdXRNZXRyaWMoXG4gICAgICAgIE1ldHJpY05hbWUuRk9VTkRfTElDRU5TRV9GSUxFLFxuICAgICAgICBsaWNlbnNlVGV4dCAhPSBudWxsID8gMSA6IDAsXG4gICAgICAgIFVuaXQuQ291bnQsXG4gICAgICApO1xuXG4gICAgICAvLyBBZGQgY3VzdG9tIGxpbmtzIGNvbnRlbnQgdG8gbWV0ZGF0YSBmb3IgZGlzcGxheSBvbiB0aGUgZnJvbnRlbmRcbiAgICAgIGNvbnN0IGFsbG93ZWRMaW5rczogUGFja2FnZUxpbmtDb25maWdbXSA9IEpTT04ucGFyc2UoUEFDS0FHRV9MSU5LUyk7XG5cbiAgICAgIGNvbnN0IHBhY2thZ2VMaW5rcyA9IGFsbG93ZWRMaW5rcy5yZWR1Y2UoKGFjY3VtLCB7IGNvbmZpZ0tleSwgYWxsb3dlZERvbWFpbnMgfSkgPT4ge1xuICAgICAgICBjb25zdCBwa2dWYWx1ZSA9IGNvbnN0cnVjdEh1Yj8ucGFja2FnZUxpbmtzW2NvbmZpZ0tleV07XG5cbiAgICAgICAgaWYgKCFwa2dWYWx1ZSkge1xuICAgICAgICAgIHJldHVybiBhY2N1bTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIGNoZWNrIGlmIHZhbHVlIGlzIGluIGFsbG93ZWQgZG9tYWlucyBsaXN0XG4gICAgICAgIGNvbnN0IHVybCA9IG5ldyBVUkwocGtnVmFsdWUpO1xuICAgICAgICBpZiAoYWxsb3dlZERvbWFpbnM/Lmxlbmd0aCAmJiAhYWxsb3dlZERvbWFpbnMuaW5jbHVkZXModXJsLmhvc3QpKSB7XG4gICAgICAgICAgcmV0dXJuIGFjY3VtO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gaWYgbm8gYWxsb3cgbGlzdCBpcyBwcm92aWRlZFxuICAgICAgICByZXR1cm4geyAuLi5hY2N1bSwgW2NvbmZpZ0tleV06IHBrZ1ZhbHVlIH07XG4gICAgICB9LCB7fSk7XG5cbiAgICAgIGNvbnN0IG1ldGFkYXRhID0ge1xuICAgICAgICBkYXRlOiBwYXlsb2FkLnRpbWUsXG4gICAgICAgIGxpY2Vuc2VUZXh0OiBsaWNlbnNlVGV4dD8udG9TdHJpbmcoJ3V0Zi04JyksXG4gICAgICAgIHBhY2thZ2VMaW5rcyxcbiAgICAgIH07XG5cbiAgICAgIGNvbnN0IHsgYXNzZW1ibHlLZXksIG1ldGFkYXRhS2V5LCBwYWNrYWdlS2V5IH0gPSBjb25zdGFudHMuZ2V0T2JqZWN0S2V5cyhcbiAgICAgICAgcGFja2FnZU5hbWUsXG4gICAgICAgIHBhY2thZ2VWZXJzaW9uLFxuICAgICAgKTtcbiAgICAgIGNvbnNvbGUubG9nKGBXcml0aW5nIGFzc2VtYmx5IGF0ICR7YXNzZW1ibHlLZXl9YCk7XG4gICAgICBjb25zb2xlLmxvZyhgV3JpdGluZyBwYWNrYWdlIGF0ICAke3BhY2thZ2VLZXl9YCk7XG4gICAgICBjb25zb2xlLmxvZyhgV3JpdGluZyBtZXRhZGF0YSBhdCAgJHttZXRhZGF0YUtleX1gKTtcblxuICAgICAgLy8gd2UgdXBsb2FkIHRoZSBtZXRhZGF0YSBmaWxlIGZpcnN0IGJlY2F1c2UgdGhlIGNhdGFsb2cgYnVpbGRlciBkZXBlbmRzIG9uXG4gICAgICAvLyBpdCBhbmQgaXMgdHJpZ2dlcmVkIGJ5IHRoZSBhc3NlbWJseSBmaWxlIHVwbG9hZC5cbiAgICAgIGNvbnNvbGUubG9nKFxuICAgICAgICBgJHtwYWNrYWdlTmFtZX1AJHtwYWNrYWdlVmVyc2lvbn0gfCBVcGxvYWRpbmcgcGFja2FnZSBhbmQgbWV0YWRhdGEgZmlsZXNgLFxuICAgICAgKTtcbiAgICAgIGNvbnN0IFtwa2csIHN0b3JlZE1ldGFkYXRhXSA9IGF3YWl0IFByb21pc2UuYWxsKFtcbiAgICAgICAgYXdzXG4gICAgICAgICAgLnMzKClcbiAgICAgICAgICAucHV0T2JqZWN0KHtcbiAgICAgICAgICAgIEJ1Y2tldDogQlVDS0VUX05BTUUsXG4gICAgICAgICAgICBLZXk6IHBhY2thZ2VLZXksXG4gICAgICAgICAgICBCb2R5OiB0YXJiYWxsLkJvZHksXG4gICAgICAgICAgICBDYWNoZUNvbnRyb2w6ICdwdWJsaWMnLFxuICAgICAgICAgICAgQ29udGVudFR5cGU6ICdhcHBsaWNhdGlvbi9vY3RldC1zdHJlYW0nLFxuICAgICAgICAgICAgTWV0YWRhdGE6IHtcbiAgICAgICAgICAgICAgJ0xhbWJkYS1Mb2ctR3JvdXAnOiBjb250ZXh0LmxvZ0dyb3VwTmFtZSxcbiAgICAgICAgICAgICAgJ0xhbWJkYS1Mb2ctU3RyZWFtJzogY29udGV4dC5sb2dTdHJlYW1OYW1lLFxuICAgICAgICAgICAgICAnTGFtYmRhLVJ1bi1JZCc6IGNvbnRleHQuYXdzUmVxdWVzdElkLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9KVxuICAgICAgICAgIC5wcm9taXNlKCksXG4gICAgICAgIGF3c1xuICAgICAgICAgIC5zMygpXG4gICAgICAgICAgLnB1dE9iamVjdCh7XG4gICAgICAgICAgICBCdWNrZXQ6IEJVQ0tFVF9OQU1FLFxuICAgICAgICAgICAgS2V5OiBtZXRhZGF0YUtleSxcbiAgICAgICAgICAgIEJvZHk6IEpTT04uc3RyaW5naWZ5KG1ldGFkYXRhKSxcbiAgICAgICAgICAgIENhY2hlQ29udHJvbDogJ3B1YmxpYycsXG4gICAgICAgICAgICBDb250ZW50VHlwZTogJ2FwcGxpY2F0aW9uL2pzb24nLFxuICAgICAgICAgICAgTWV0YWRhdGE6IHtcbiAgICAgICAgICAgICAgJ0xhbWJkYS1Mb2ctR3JvdXAnOiBjb250ZXh0LmxvZ0dyb3VwTmFtZSxcbiAgICAgICAgICAgICAgJ0xhbWJkYS1Mb2ctU3RyZWFtJzogY29udGV4dC5sb2dTdHJlYW1OYW1lLFxuICAgICAgICAgICAgICAnTGFtYmRhLVJ1bi1JZCc6IGNvbnRleHQuYXdzUmVxdWVzdElkLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9KVxuICAgICAgICAgIC5wcm9taXNlKCksXG4gICAgICBdKTtcblxuICAgICAgLy8gbm93IHdlIGNhbiB1cGxvYWQgdGhlIGFzc2VtYmx5LlxuICAgICAgY29uc29sZS5sb2coYCR7cGFja2FnZU5hbWV9QCR7cGFja2FnZVZlcnNpb259IHwgVXBsb2FkaW5nIGFzc2VtYmx5IGZpbGVgKTtcbiAgICAgIGNvbnN0IGFzc2VtYmx5ID0gYXdhaXQgYXdzXG4gICAgICAgIC5zMygpXG4gICAgICAgIC5wdXRPYmplY3Qoe1xuICAgICAgICAgIEJ1Y2tldDogQlVDS0VUX05BTUUsXG4gICAgICAgICAgS2V5OiBhc3NlbWJseUtleSxcbiAgICAgICAgICBCb2R5OiBkb3RKc2lpLFxuICAgICAgICAgIENhY2hlQ29udHJvbDogJ3B1YmxpYycsXG4gICAgICAgICAgQ29udGVudFR5cGU6ICdhcHBsaWNhdGlvbi9qc29uJyxcbiAgICAgICAgICBNZXRhZGF0YToge1xuICAgICAgICAgICAgJ0xhbWJkYS1Mb2ctR3JvdXAnOiBjb250ZXh0LmxvZ0dyb3VwTmFtZSxcbiAgICAgICAgICAgICdMYW1iZGEtTG9nLVN0cmVhbSc6IGNvbnRleHQubG9nU3RyZWFtTmFtZSxcbiAgICAgICAgICAgICdMYW1iZGEtUnVuLUlkJzogY29udGV4dC5hd3NSZXF1ZXN0SWQsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSlcbiAgICAgICAgLnByb21pc2UoKTtcblxuICAgICAgY29uc3QgY3JlYXRlZDogU3RhdGVNYWNoaW5lSW5wdXQgPSB7XG4gICAgICAgIGJ1Y2tldDogQlVDS0VUX05BTUUsXG4gICAgICAgIGFzc2VtYmx5OiB7XG4gICAgICAgICAga2V5OiBhc3NlbWJseUtleSxcbiAgICAgICAgICB2ZXJzaW9uSWQ6IGFzc2VtYmx5LlZlcnNpb25JZCxcbiAgICAgICAgfSxcbiAgICAgICAgcGFja2FnZToge1xuICAgICAgICAgIGtleTogcGFja2FnZUtleSxcbiAgICAgICAgICB2ZXJzaW9uSWQ6IHBrZy5WZXJzaW9uSWQsXG4gICAgICAgIH0sXG4gICAgICAgIG1ldGFkYXRhOiB7XG4gICAgICAgICAga2V5OiBtZXRhZGF0YUtleSxcbiAgICAgICAgICB2ZXJzaW9uSWQ6IHN0b3JlZE1ldGFkYXRhLlZlcnNpb25JZCxcbiAgICAgICAgfSxcbiAgICAgIH07XG4gICAgICBjb25zb2xlLmxvZyhgQ3JlYXRlZCBvYmplY3RzOiAke0pTT04uc3RyaW5naWZ5KGNyZWF0ZWQsIG51bGwsIDIpfWApO1xuXG4gICAgICBjb25zdCBzZm4gPSBhd2FpdCBhd3NcbiAgICAgICAgLnN0ZXBGdW5jdGlvbnMoKVxuICAgICAgICAuc3RhcnRFeGVjdXRpb24oe1xuICAgICAgICAgIGlucHV0OiBKU09OLnN0cmluZ2lmeShjcmVhdGVkKSxcbiAgICAgICAgICBuYW1lOiBzZm5FeGVjdXRpb25OYW1lRnJvbVBhcnRzKFxuICAgICAgICAgICAgcGFja2FnZU5hbWUsXG4gICAgICAgICAgICBgdiR7cGFja2FnZVZlcnNpb259YCxcbiAgICAgICAgICAgIGNvbnRleHQuYXdzUmVxdWVzdElkLFxuICAgICAgICAgICksXG4gICAgICAgICAgc3RhdGVNYWNoaW5lQXJuOiBTVEFURV9NQUNISU5FX0FSTixcbiAgICAgICAgfSlcbiAgICAgICAgLnByb21pc2UoKTtcbiAgICAgIGNvbnNvbGUubG9nKGBTdGFydGVkIFN0YXRlTWFjaGluZSBleGVjdXRpb246ICR7c2ZuLmV4ZWN1dGlvbkFybn1gKTtcbiAgICAgIHJlc3VsdC5wdXNoKHNmbi5leGVjdXRpb25Bcm4pO1xuICAgIH1cblxuICAgIHJldHVybiByZXN1bHQ7XG4gIH0sXG4pO1xuXG4vKipcbiAqIENoZWNrcyB3aGV0aGVyIHRoZSBwcm92aWRlZCBmaWxlIG5hbWUgY29ycmVzcG9uZHMgdG8gYSBsaWNlbnNlIGZpbGUgb3Igbm90LlxuICpcbiAqIEBwYXJhbSBmaWxlTmFtZSB0aGUgZmlsZSBuYW1lIHRvIGJlIGNoZWNrZWQuXG4gKlxuICogQHJldHVybnMgYHRydWVgIElJRiB0aGUgZmlsZSBpcyBuYW1lZCBMSUNFTlNFIGFuZCBoYXMgdGhlIC5NRCBvciAuVFhUXG4gKiAgICAgICAgICBleHRlbnNpb24sIG9yIG5vIGV4dGVuc2lvbiBhdCBhbGwuIFRoZSB0ZXN0IGlzIGNhc2UtaW5zZW5zaXRpdmUuXG4gKi9cbmZ1bmN0aW9uIGlzTGljZW5zZUZpbGUoZmlsZU5hbWU6IHN0cmluZyk6IGJvb2xlYW4ge1xuICBjb25zdCBleHQgPSBleHRuYW1lKGZpbGVOYW1lKTtcbiAgY29uc3QgcG9zc2libGVFeHRlbnNpb25zID0gbmV3IFNldChbJycsICcubWQnLCAnLnR4dCddKTtcbiAgcmV0dXJuIChcbiAgICBwb3NzaWJsZUV4dGVuc2lvbnMuaGFzKGV4dC50b0xvd2VyQ2FzZSgpKSAmJlxuICAgIGJhc2VuYW1lKGZpbGVOYW1lLCBleHQpLnRvVXBwZXJDYXNlKCkgPT09ICdMSUNFTlNFJ1xuICApO1xufVxuXG4vKipcbiAqIENyZWF0ZXMgYSBTdGVwRnVuY3Rpb24gZXhlY3V0aW9uIHJlcXVlc3QgbmFtZSBiYXNlZCBvbiB0aGUgcHJvdmlkZWQgcGFydHMuXG4gKiBUaGUgcmVzdWx0IGlzIGd1YXJhbnRlZWQgdG8gYmUgODAgY2hhcmFjdGVycyBvciBsZXNzIGFuZCB0byBjb250YWluIG9ubHlcbiAqIGNoYXJhY3RlcnMgdGhhdCBhcmUgdmFsaWQgZm9yIGEgU3RlcEZ1bmN0aW9uIGV4ZWN1dGlvbiByZXF1ZXN0IG5hbWUgZm9yIHdoaWNoXG4gKiBDbG91ZFdhdGNoIExvZ2dpbmcgY2FuIGJlIGVuYWJsZWQuIFRoZSByZXN1bHRpbmcgbmFtZSBpcyB2ZXJ5IGxpa2VseSB0b1xuICogYmUgdW5pcXVlIGZvciBhIGdpdmVuIGlucHV0LlxuICovXG5mdW5jdGlvbiBzZm5FeGVjdXRpb25OYW1lRnJvbVBhcnRzKFxuICBmaXJzdDogc3RyaW5nLFxuICAuLi5yZXN0OiByZWFkb25seSBzdHJpbmdbXVxuKTogc3RyaW5nIHtcbiAgY29uc3QgcGFydHMgPSBbZmlyc3QsIC4uLnJlc3RdO1xuICBjb25zdCBuYW1lID0gcGFydHNcbiAgICAubWFwKChwYXJ0KSA9PiBwYXJ0LnJlcGxhY2UoL1teYS16MC05Xy1dKy9naSwgJ18nKSlcbiAgICAuam9pbignXycpXG4gICAgLnJlcGxhY2UoL15fL2csICcnKVxuICAgIC5yZXBsYWNlKC9fezIsfS9nLCAnXycpO1xuICBpZiAobmFtZS5sZW5ndGggPD0gODApIHtcbiAgICByZXR1cm4gbmFtZTtcbiAgfVxuICBjb25zdCBzdWZmaXggPSBjcmVhdGVIYXNoKCdzaGEyNTYnKVxuICAgIC8vIFRoZSBoYXNoIGlzIGNvbXB1dGVkIGJhc2VkIG9uIGlucHV0IGFyZ3VtZW50cywgdG8gbWF4aW1pemUgdW5pY2l0eVxuICAgIC51cGRhdGUocGFydHMuam9pbignXycpKVxuICAgIC5kaWdlc3QoJ2hleCcpXG4gICAgLnN1YnN0cmluZygwLCA2KTtcbiAgcmV0dXJuIGAke25hbWUuc3Vic3RyaW5nKDAsIDgwIC0gc3VmZml4Lmxlbmd0aCAtIDEpfV8ke3N1ZmZpeH1gO1xufVxuIl19