"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackendDashboard = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const language_1 = require("./backend/shared/language");
const deep_link_1 = require("./deep-link");
const metric_utils_1 = require("./metric-utils");
class BackendDashboard extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        new aws_cloudwatch_1.Dashboard(this, 'Resource', {
            dashboardName: props.dashboardName,
            periodOverride: aws_cloudwatch_1.PeriodOverride.INHERIT,
            widgets: [
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Catalog Overview',
                            '',
                            `[button:Package Data](${deep_link_1.s3ObjectUrl(props.packageData)})`,
                            `[button:Catalog Builder](${deep_link_1.lambdaFunctionUrl(props.orchestration.catalogBuilder)})`,
                            `[button:Inventory Canary](${deep_link_1.lambdaFunctionUrl(props.inventory.function)})`,
                            `[button:Search Canary Log Group](${deep_link_1.lambdaSearchLogGroupUrl(props.inventory.function)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Catalog Size',
                        left: [
                            props.inventory.metricSubmoduleCount({ label: 'Submodules' }),
                            props.inventory.metricPackageVersionCount({ label: 'Package Versions' }),
                            props.inventory.metricPackageMajorCount({ label: 'Package Majors' }),
                            props.inventory.metricPackageCount({ label: 'Packages' }),
                        ],
                        leftYAxis: { min: 0 },
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Catalog Issues',
                        left: [
                            props.inventory.metricUnknownObjectCount({ label: 'Unknown' }),
                            props.inventory.metricMissingAssemblyCount({ label: 'Missing Assembly' }),
                            props.inventory.metricMissingPackageMetadataCount({ label: 'Missing Metadata' }),
                            props.inventory.metricMissingPackageTarballCount({ label: 'Missing Tarball' }),
                        ],
                        leftYAxis: { min: 0 },
                    }),
                ],
                ...this.catalogOverviewLanguageSections(props.inventory),
                ...renderPackageSourcesWidgets(props.packageSources),
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Ingestion Function',
                            '',
                            `[button:Ingestion Function](${deep_link_1.lambdaFunctionUrl(props.ingestion.function)})`,
                            `[button:Search Log Group](${deep_link_1.lambdaSearchLogGroupUrl(props.ingestion.function)})`,
                            `[button:DLQ](${deep_link_1.sqsQueueUrl(props.ingestion.deadLetterQueue)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Function Health',
                        left: [
                            metric_utils_1.fillMetric(props.ingestion.function.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(props.ingestion.function.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Input Queue',
                        left: [
                            props.ingestion.queue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages', period: core_1.Duration.minutes(1) }),
                            props.ingestion.queue.metricApproximateNumberOfMessagesNotVisible({ label: 'Hidden Messages', period: core_1.Duration.minutes(1) }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.ingestion.queue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age', period: core_1.Duration.minutes(1) }),
                        ],
                        rightAnnotations: [{
                                color: '#ffa500',
                                label: '10 Minutes',
                                value: core_1.Duration.minutes(10).toSeconds(),
                            }],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Input Quality',
                        left: [
                            metric_utils_1.fillMetric(props.ingestion.metricInvalidAssembly({ label: 'Invalid Assemblies' })),
                            metric_utils_1.fillMetric(props.ingestion.metricInvalidTarball({ label: 'Invalid Tarball' })),
                            metric_utils_1.fillMetric(props.ingestion.metricIneligibleLicense({ label: 'Ineligible License' })),
                            metric_utils_1.fillMetric(props.ingestion.metricMismatchedIdentityRejections({ label: 'Mismatched Identity' })),
                            metric_utils_1.fillMetric(props.ingestion.metricFoundLicenseFile({ label: 'Found License file' })),
                        ],
                        leftYAxis: { label: 'Count', min: 0, showUnits: false },
                        stacked: true,
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letters',
                        left: [
                            props.ingestion.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }),
                            props.ingestion.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.ingestion.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Orchestration',
                            '',
                            `[button:State Machine](${deep_link_1.stateMachineUrl(props.orchestration.stateMachine)})`,
                            `[button:DLQ](${deep_link_1.sqsQueueUrl(props.orchestration.deadLetterQueue)})`,
                            `[button:Redrive DLQ](${deep_link_1.lambdaFunctionUrl(props.orchestration.redriveFunction)})`,
                            `[button:Reprocess](${deep_link_1.lambdaFunctionUrl(props.orchestration.reprocessAllFunction)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'State Machine Executions',
                        left: [
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricStarted({ label: 'Started' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricSucceeded({ label: 'Succeeded' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricAborted({ label: 'Aborted' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricFailed({ label: 'Failed' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricThrottled({ label: 'Throttled' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricTimedOut({ label: 'Timed Out' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.stateMachine.metricTime({ label: 'Duration' }),
                        ],
                        rightYAxis: { min: 0 },
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letter Queue',
                        left: [
                            props.orchestration.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }),
                            props.orchestration.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                // deny list
                // ----------------------------------------------
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Deny List',
                            '',
                            `[button:Deny List Object](${deep_link_1.s3ObjectUrl(props.denyList.bucket, props.denyList.objectKey)})`,
                            `[button:Prune Function](${deep_link_1.lambdaFunctionUrl(props.denyList.prune.pruneHandler)})`,
                            `[button:Prune Logs](${deep_link_1.lambdaSearchLogGroupUrl(props.denyList.prune.pruneHandler)})`,
                            `[button:Delete Queue](${deep_link_1.sqsQueueUrl(props.denyList.prune.queue)})`,
                            `[button:Delete Logs](${deep_link_1.lambdaSearchLogGroupUrl(props.denyList.prune.deleteHandler)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Deny List',
                        left: [
                            metric_utils_1.fillMetric(props.denyList.metricDenyListRules({ label: 'Rules' }), 'REPEAT'),
                            props.denyList.prune.queue.metricNumberOfMessagesDeleted({ label: 'Deleted Files' }),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Prune Function Health',
                        left: [
                            metric_utils_1.fillMetric(props.denyList.prune.pruneHandler.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(props.denyList.prune.pruneHandler.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                ],
            ],
        });
    }
    *catalogOverviewLanguageSections(inventory) {
        yield [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: '# Documentation Generation',
            }),
        ];
        for (const language of language_1.DocumentationLanguage.ALL) {
            yield [
                new aws_cloudwatch_1.TextWidget({
                    height: 1,
                    width: 24,
                    markdown: `## Language: ${language.toString()}`,
                }),
            ];
            yield [
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Versions',
                    left: [
                        inventory.metricSupportedPackageVersionCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedPackageVersionCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingPackageVersionCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    view: aws_cloudwatch_1.GraphWidgetView.PIE,
                }),
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Versions',
                    left: [
                        inventory.metricSupportedPackageVersionCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedPackageVersionCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingPackageVersionCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    stacked: true,
                }),
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Version Submodules',
                    left: [
                        inventory.metricSupportedSubmoduleCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedSubmoduleCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingSubmoduleCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    view: aws_cloudwatch_1.GraphWidgetView.PIE,
                }),
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Version Submodules',
                    left: [
                        inventory.metricSupportedSubmoduleCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedSubmoduleCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingSubmoduleCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    stacked: true,
                }),
            ];
        }
    }
}
exports.BackendDashboard = BackendDashboard;
function* renderPackageSourcesWidgets(packageSources) {
    var _a;
    for (const packageSource of packageSources) {
        yield [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: [
                    `# ${packageSource.name}`,
                    '',
                    ...((_a = packageSource.links) !== null && _a !== void 0 ? _a : [])
                        .map(({ name, primary, url }) => `[${primary ? 'button:primary' : 'button'}:${name}](${url})`),
                ].join('\n'),
            }),
        ];
        yield* packageSource.dashboardWidgets;
    }
    return;
}
//# sourceMappingURL=data:application/json;base64,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