"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const assert_1 = require("@aws-cdk/assert");
const cf = require("@aws-cdk/aws-cloudfront");
const aws_cloudfront_origins_1 = require("@aws-cdk/aws-cloudfront-origins");
const cdk = require("@aws-cdk/core");
const extensions = require("../extensions");
test('minimal usage', () => {
    // GIVEN
    const app = new cdk.App();
    const stack = new cdk.Stack(app, 'demo-stack');
    // WHEN
    // create a cloudfront distribution with an extension (L@E)
    const convertQueryStringProsp = { args: ['language', 'name'] };
    const convertQueryString = new extensions.ConvertQueryString(stack, 'ConvertQueryStringDemo', convertQueryStringProsp);
    // A CloudFront distribution
    const cloudFrontDistribution = new cf.Distribution(stack, 'CloudFrontDistribution', {
        defaultBehavior: {
            origin: new aws_cloudfront_origins_1.HttpOrigin('postman-echo.com', {
                httpPort: 80,
                originPath: '/get',
                originSslProtocols: [cf.OriginSslPolicy.TLS_V1],
                keepaliveTimeout: cdk.Duration.seconds(10),
                protocolPolicy: cf.OriginProtocolPolicy.HTTP_ONLY,
                readTimeout: cdk.Duration.seconds(10),
            }),
            edgeLambdas: [convertQueryString],
            cachePolicy: new cf.CachePolicy(stack, 'DefaultCachePolicy', {
                cachePolicyName: 'ConvertQueryString-Cache-Policy',
                queryStringBehavior: cf.CacheQueryStringBehavior.all(),
            }),
            originRequestPolicy: new cf.OriginRequestPolicy(stack, 'RequestPolicy', {
                originRequestPolicyName: 'ConvertQueryString-Request-Policy',
                queryStringBehavior: cf.OriginRequestQueryStringBehavior.all(),
                headerBehavior: cf.OriginRequestHeaderBehavior.all(),
                comment: 'just for demonstration.',
            }),
        },
        comment: 'The CloudFront distribution based on the custom origin',
        priceClass: cf.PriceClass.PRICE_CLASS_200,
    });
    new cdk.CfnOutput(stack, 'DistributionDomainName', {
        value: cloudFrontDistribution.distributionDomainName,
    });
    // THEN
    expect(assert_1.SynthUtils.synthesize(stack).template).toMatchSnapshot();
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Comment: 'The CloudFront distribution based on the custom origin',
            DefaultCacheBehavior: {
                LambdaFunctionAssociations: [
                    {
                        EventType: 'origin-request',
                        IncludeBody: false,
                        LambdaFunctionARN: {
                            Ref: 'ConvertQueryStringFuncCurrentVersion4FB2758683a3f08fef38e4df5b692cceeaaf7608',
                        },
                    },
                ],
                ViewerProtocolPolicy: 'allow-all',
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        HTTPPort: 80,
                        OriginKeepaliveTimeout: 10,
                        OriginProtocolPolicy: 'http-only',
                        OriginReadTimeout: 10,
                        OriginSSLProtocols: [
                            'TLSv1',
                        ],
                    },
                    DomainName: 'postman-echo.com',
                    Id: 'demostackCloudFrontDistributionOrigin15405BC3B',
                    OriginPath: '/get',
                },
            ],
            PriceClass: 'PriceClass_200',
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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