import requests
from typing import Optional
from vesselapi.models import operations
from . import utils

class Events:
    _client: requests.Session
    _security_client: requests.Session
    _server_url: str
    _language: str
    _sdk_version: str
    _gen_version: str

    def __init__(self, client: requests.Session, security_client: requests.Session, server_url: str, language: str, sdk_version: str, gen_version: str) -> None:
        self._client = client
        self._security_client = security_client
        self._server_url = server_url
        self._language = language
        self._sdk_version = sdk_version
        self._gen_version = gen_version

    
    def batch(self, request: operations.GetBatchCrmEventRequest) -> operations.GetBatchCrmEventResponse:
        r"""Get Batch Events
        Retrieve Events by a set of Id's
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix("/") + "/crm/event/batch"
        
        query_params = utils.get_query_params(request.query_params)
        
        client = self._security_client
        
        r = client.request("GET", url, params=query_params)
        content_type = r.headers.get("Content-Type")

        res = operations.GetBatchCrmEventResponse(status_code=r.status_code, content_type=content_type)
        
        if r.status_code == 200:
            if utils.match_content_type(content_type, "application/json"):
                out = utils.unmarshal_json(r.text, Optional[operations.GetBatchCrmEventResponseBody])
                res.response_body = out

        return res

    
    def create(self, request: operations.PostCrmEventRequest) -> operations.PostCrmEventResponse:
        r"""Create Event
        Create a new Event.
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix("/") + "/crm/event"
        
        headers = {}
        req_content_type, data, json, files = utils.serialize_request_body(request)
        if req_content_type != "multipart/form-data" and req_content_type != "multipart/mixed":
            headers["content-type"] = req_content_type
        
        client = self._security_client
        
        r = client.request("POST", url, data=data, json=json, files=files, headers=headers)
        content_type = r.headers.get("Content-Type")

        res = operations.PostCrmEventResponse(status_code=r.status_code, content_type=content_type)
        
        if r.status_code == 200:
            if utils.match_content_type(content_type, "application/json"):
                out = utils.unmarshal_json(r.text, Optional[operations.PostCrmEventResponseBody])
                res.response_body = out

        return res

    
    def details(self, request: operations.GetDetailsCrmEventRequest) -> operations.GetDetailsCrmEventResponse:
        r"""Get Event Details
        Get details about all events. 
        Details include the type, possible values, and other meta data about the fields.
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix("/") + "/crm/event/details"
        
        query_params = utils.get_query_params(request.query_params)
        
        client = self._security_client
        
        r = client.request("GET", url, params=query_params)
        content_type = r.headers.get("Content-Type")

        res = operations.GetDetailsCrmEventResponse(status_code=r.status_code, content_type=content_type)
        
        if r.status_code == 200:
            if utils.match_content_type(content_type, "application/json"):
                out = utils.unmarshal_json(r.text, Optional[operations.GetDetailsCrmEventResponseBody])
                res.response_body = out

        return res

    
    def find(self, request: operations.GetOneCrmEventRequest) -> operations.GetOneCrmEventResponse:
        r"""Get Event
        Retrieve a single Event by Id
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix("/") + "/crm/event"
        
        query_params = utils.get_query_params(request.query_params)
        
        client = self._security_client
        
        r = client.request("GET", url, params=query_params)
        content_type = r.headers.get("Content-Type")

        res = operations.GetOneCrmEventResponse(status_code=r.status_code, content_type=content_type)
        
        if r.status_code == 200:
            if utils.match_content_type(content_type, "application/json"):
                out = utils.unmarshal_json(r.text, Optional[operations.GetOneCrmEventResponseBody])
                res.response_body = out

        return res

    
    def list(self, request: operations.GetAllCrmEventsRequest) -> operations.GetAllCrmEventsResponse:
        r"""Get All Events
        Retrieve all Events
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix("/") + "/crm/events"
        
        query_params = utils.get_query_params(request.query_params)
        
        client = self._security_client
        
        r = client.request("GET", url, params=query_params)
        content_type = r.headers.get("Content-Type")

        res = operations.GetAllCrmEventsResponse(status_code=r.status_code, content_type=content_type)
        
        if r.status_code == 200:
            if utils.match_content_type(content_type, "application/json"):
                out = utils.unmarshal_json(r.text, Optional[operations.GetAllCrmEventsResponseBody])
                res.response_body = out

        return res

    
    def update(self, request: operations.PutCrmEventRequest) -> operations.PutCrmEventResponse:
        r"""Update Event
        Update an existing Event by Id
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix("/") + "/crm/event"
        
        headers = {}
        req_content_type, data, json, files = utils.serialize_request_body(request)
        if req_content_type != "multipart/form-data" and req_content_type != "multipart/mixed":
            headers["content-type"] = req_content_type
        
        client = self._security_client
        
        r = client.request("PATCH", url, data=data, json=json, files=files, headers=headers)
        content_type = r.headers.get("Content-Type")

        res = operations.PutCrmEventResponse(status_code=r.status_code, content_type=content_type)
        
        if r.status_code == 200:
            if utils.match_content_type(content_type, "application/json"):
                out = utils.unmarshal_json(r.text, Optional[operations.PutCrmEventResponseBody])
                res.response_body = out

        return res

    