# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['quaternionic']

package_data = \
{'': ['*']}

install_requires = \
['numba>=0.50,<0.51', 'numpy>=1.13,<2.0', 'scipy>=1.0,<2.0']

extras_require = \
{':python_version < "3.8"': ['importlib-metadata>=1.0,<2.0'],
 'mkdocs': ['mkdocs>=1.1.2,<2.0.0'],
 'mktheapidocs': ['mktheapidocs[plugin]>=0.2.0,<0.3.0']}

setup_kwargs = {
    'name': 'quaternionic',
    'version': '0.1.0',
    'description': 'Interpret numpy arrays as quaternionic arrays with numba acceleration',
    'long_description': '# Quaternions by way of numpy arrays\n\nThis module subclasses numpy\'s array type, interpreting the array as an array of quaternions, and\naccelerating the algebra using numba.  There is also basic initial support for symbolic manipulation\nof quaternions by creating quaternionic arrays with sympy symbols as elements, though this is a work\nin progress.\n\nThis package has evolved from the [quaternion](https://github.com/moble/quaternion) package, which\nadds a quaternion dtype directly to numpy.  In many ways, that is a much better approach because\ndtypes are built in to numpy, making it more robust than this package.  However, that approach has\nits own limitations, including that it is harder to maintain, and requires much of the code to be\nwritten in C, which also makes it harder to distribute.  This package is written entirely in python\ncode, but should actually have comparable performance because it is compiled by numba.  Moreover,\nbecause the core code is written in pure python, it is reusable for purposes other than the core\npurpose of this package, which is to provide the numeric array type.\n\n\n# Installation\n\nBecause this package is pure python code, it can be installed with the simplest tools.  In\nparticular, you can just run\n\n```bash\npip install quaternionic\n```\n\nFor development work, the best current option is [poetry](https://python-poetry.org/).  From the\ntop-level directory, run `poetry install` or just `poetry run <some command>`.\n\n# Usage\n\nAny numpy array `a` with a last axis of size 4 (and float dtype) can be reinterpreted as a\nquaternionic array with `quaternionic.array(a)`:\n\n```python\nimport numpy as np\nimport quaternionic\n\na = 1.0 - np.random.rand(17, 3, 4)  # Just some random numbers; last dimension is 4\nq1 = quaternionic.array(a)  # Reinterpret an existing array\nq2 = quaternionic.array([1.2, 2.3, 3.4, 4.5])  # Create a new array\n```\n\nHere, the original array `a` will still exist just as it was, and will behave just as a normal numpy\narray — including changing its values (which will change the values in `q1`), slicing, math, etc.\nHowever, `q1` will be another\n["view"](https://numpy.org/doc/stable/reference/generated/numpy.ndarray.view.html) into the same\ndata.  Operations on `q1` will be quaternionic.  For example, whereas `1/a` returns the element-wise\ninverse of each float in the array, `1/q1` returns the *quaternionic* inverse of each quaternion.\nSimilarly, if you multiply two quaternionic arrays, their product will be computed with the usual\nquaternion multiplication, rather than element-wise multiplication of floats as numpy usually\nperforms.\n\nAll the usual quaternion operations are available, including\n\n  * Addition `q1 + q2`\n  * Subtraction `q1 - q2`\n  * Multiplication `q1 * q2`\n  * Division `q1 / q2`\n  * Scalar multiplication `q1 * s == s * q1`\n  * Scalar division `q1 / s != s / q1`\n  * Exponential `np.exp(q1)`\n  * Logarithm `np.log(q1)`\n  * Square-root `np.sqrt(q1)`\n\nAll numpy [ufuncs](https://numpy.org/doc/stable/reference/ufuncs.html) that make sense for\nquaternions are supported.  When the arrays have different shapes, the usual numpy\n[broadcasting](https://numpy.org/doc/stable/user/basics.broadcasting.html) rules take effect.\n\nIn addition to the basic numpy array features, we also have a number of extra properties that are\nparticularly useful for quaternions, including\n\n  * Methods to extract and/or set components\n    * w, x, y, z\n    * i, j, k (equivalent to x, y, z)\n    * scalar, vector (equivalent to w, [x, y, z])\n    * real, imag (equivalent to scalar, vector)\n  * Methods related to norms\n    * abs (square-root of sum of squares of components)\n    * norm (sum of squares of components)\n    * modulus, magnitude (equal to abs)\n    * absolute\\_square, abs2, mag2, squared_norm (equal to norm)\n    * normalized\n    * conjugate, conj\n    * inverse\n  * Methods related to array infrastructure\n    * ndarray (the numpy array underlying the quaternionic array)\n    * flattened (all dimensions but last are flattened into one)\n    * iterator (iterate over all quaternions)\n    \nNote that this package makes a distinction between `abs` and `norm` — the latter being the square of\nthe former.  This choice agrees with the [Boost library\'s implementation of\nquaternions](https://www.boost.org/doc/libs/1_74_0/libs/math/doc/html/math_toolkit/value_op.html),\nas well as this package\'s forerunner\n[quaternion](https://github.com/moble/quaternion/blob/99913120b1b2a8a5eb7769c29ee760a236d40880/quaternion.h#L115-L120).\nThis also agrees with the corresponding functions on the [C++ standard library\'s complex\nnumbers](http://www.cplusplus.com/reference/complex/norm/).  Because this may be confusing, a number\nof aliases are also provided that may be less confusing.  For example, some people find the pair\n`abs` and `abs2` to be more sensible.\n\nThis package does not specialize to *unit* quaternions, since it is usually better for numerical\npurposes not to do so.  For example, whereas rotation of a vector `v` by a quaternion is usually\nimplemented as `R * v * np.conjugate(R)`, it is usually better to drop the assumption that the\nquaternion has unit magnitude and implement rotation as `R * v * np.reciprocal(R)`.  That is what\nthis package does by default whenever rotations are involved.\n\nAlthough this package does not specialize to unit quaternions, there are several converters to and\nfrom other representations of rotations, including\n\n   * to/from\\_rotation\\_matrix\n   * to\\_transformation\\_matrix (for non-unit quaternions)\n   * to/from\\_axis\\_angle representation\n   * to/from\\_euler\\_angles (though using Euler angles is almost always a bad idea)\n   * to/from\\_spherical\\_coordinates\n   * to/from\\_angular\\_velocity\n\nNote that the last item relates to quaternion-valued functions of time.  Converting to an angular\nvelocity requires differentiation, while converting from angular velocity requires integration (as\nexplored in [this paper](https://arxiv.org/abs/1604.08139)).\n\nFor these converters, the "to" functions are properties on the individual arrays, whereas the "from"\nfunctions are "classmethod"s that take the corresponding objects as inputs.  For example, we could\nwrite\n\n```python\nq1 = quaternionic.array(np.random.rand(100, 4)).normalized\nm = q1.to_rotation_matrix\n```\n\nto obtain the matrix `m` *from* a quaternionic array `q1`.  (Here, `m` is actually a series of 100\n3x3 matrices corresponding to the 100 quaternions in `q1`.)  On the other hand, to obtain a\nquaternionic array from some matrix `m`, we would write\n\n```python\nq2 = quaternionic.array.from_rotation_matrix(m)\n```\n\nAlso note that, because the quaternions form a "double cover" of the rotation group (meaning that\nquaternions `R` and `-R` represent the same rotation), these functions are not perfect inverses of\neach other.  In this case, for example, `q1` and `q2` may have opposite signs.  We can, however,\nprove that these quaternions represent the same rotations by measuring the "distance" between the\nquaternions as rotations:\n\n```python\nnp.max(quaternionic.distance.rotation.intrinsic(q1, q2))  # Typically around 1e-15\n```\n\nThere are four types of distance measures:\n\n  * `quaternionic.distance.rotor.intrinsic`\n  * `quaternionic.distance.rotor.chordal`\n  * `quaternionic.distance.rotation.intrinsic`\n  * `quaternionic.distance.rotation.chordal`\n\nThe "rotor" distances do not account for possible differences in signs, meaning that rotor distances\ncan be large even when they represent identical rotations; the "rotation" functions just return the\nsmaller of the distance between `q1` and `q2` or the distance between `q1` and `-q2`.  The\n"intrinsic" functions measure the geodesic distance within the manifold of *unit* quaternions, and\nis somewhat slower but may be more meaningful; the "chordal" functions measure the Euclidean\ndistance in the (linear) space of all quaternions, and is faster but its precise value is not\nnecessarily as meaningful.\n\nFinally, there are also capabilities related to interpolation\n\n  * slerp (spherical linear interpolation)\n  * squad (spherical quadratic interpolation)\n\n\n\n# Related packages\n\nPackages with some quaternion features available on pypi include\n\n  * numpy-quaternion (a.k.a. quaternion, but renamed on pypi)\n  * clifford (very powerful; more general geometric algebras)\n  * pyquaternion (many features; pure python; no acceleration)\n  * quaternions (basic pure python package; no acceleration; specialized for rotations only)\n  * rowan (similar approach to this package, but no acceleration or overloading)\n  * Quaternion (minimal capabilities)\n  * scipy.spatial.transform.Rotation.as_quat (quaternion output for Rotation object)\n  * mathutils (a Blender package with python bindings)\n\nAlso note that there is some capability to do symbolic manipulations of quaternions in these packages:\n\n  * galgebra\n  * sympy.algebras.quaternion\n\n',
    'author': 'Michael Boyle',
    'author_email': 'michael.oliver.boyle@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/moble/quaternionic',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
