"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const zlib_1 = require("zlib");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const tar_stream_1 = require("tar-stream");
const client_lambda_shared_1 = require("../deny-list/client.lambda-shared");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
/**
 * Regenerates the `catalog.json` object in the configured S3 bucket.
 *
 * @param event configuration for the rebuild job. In particular, the `rebuild`
 *              property can be set to `true` in order to trigger a full (i.e:
 *              non-incremental) rebuild of the object.
 * @param context the lambda context in which this execution runs.
 *
 * @returns the information about the updated S3 object.
 */
async function handler(event, context) {
    console.log(JSON.stringify(event, null, 2));
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const packages = new Map();
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    console.log('Loading existing catalog (if present)...');
    const data = await aws.s3().getObject({ Bucket: BUCKET_NAME, Key: constants.CATALOG_KEY }).promise()
        .catch((err) => err.code !== 'NoSuchKey'
        ? Promise.reject(err)
        : Promise.resolve({ /* no data */}));
    if (data.Body) {
        console.log('Catalog found. Loading...');
        const catalog = JSON.parse(data.Body.toString('utf-8'));
        for (const info of catalog.packages) {
            if (!packages.has(info.name)) {
                packages.set(info.name, new Map());
            }
            packages.get(info.name).set(info.major, info);
        }
    }
    // If defined, the function will invoke itself again to resume the work from that key (this
    // happens only in "from scratch" or "rebuild" cases).
    let nextStartAfter;
    if (event.package) {
        if (!event.package.key.endsWith(constants.PACKAGE_KEY_SUFFIX)) {
            throw new Error(`The provided package key is invalid: ${event.package.key} does not end in ${constants.PACKAGE_KEY_SUFFIX}`);
        }
        console.log('Registering new packages...');
        // note that we intentionally don't catch errors here to let these
        // event go to the DLQ for manual inspection.
        await appendPackage(packages, event.package.key, BUCKET_NAME, denyList);
    }
    // If we don't have a package in event, then we're refreshing the catalog. This is also true if we
    // don't have a catalog body (from scratch) or if "startAfter" is set (continuation of from
    // scratch).
    if (!(event === null || event === void 0 ? void 0 : event.package) || !data.Body || event.startAfter) {
        console.log('Recreating or refreshing catalog...');
        const failures = {};
        for await (const { Key: pkgKey } of relevantObjects(BUCKET_NAME, event.startAfter)) {
            try {
                await appendPackage(packages, pkgKey, BUCKET_NAME, denyList);
            }
            catch (e) {
                failures[pkgKey] = e;
            }
            // If we're getting short on time (1 minute out of 15 left), we'll be continuing in a new
            // invocation after writing what we've done so far to S3...
            if (context.getRemainingTimeInMillis() <= 60000) {
                nextStartAfter = pkgKey;
                break;
            }
        }
        for (const [key, error] of Object.entries(failures)) {
            console.log(`Failed processing ${key}: ${error}`);
        }
        await aws_embedded_metrics_1.metricScope((metrics) => async () => {
            metrics.setDimensions();
            const failedCount = Object.keys(failures).length;
            console.log(`Marking ${failedCount} failed packages`);
            metrics.putMetric("FailedPackagesOnRecreation" /* FAILED_PACKAGES_ON_RECREATION */, failedCount, aws_embedded_metrics_1.Unit.Count);
        })();
    }
    // Build the final data package...
    console.log('Consolidating catalog...');
    const catalog = { packages: new Array(), updated: new Date().toISOString() };
    for (const majors of packages.values()) {
        for (const pkg of majors.values()) {
            catalog.packages.push(pkg);
        }
    }
    console.log(`There are now ${catalog.packages.length} registered package major versions`);
    await aws_embedded_metrics_1.metricScope((metrics) => async () => {
        metrics.setDimensions();
        metrics.putMetric("RegisteredPackagesMajorVersion" /* REGISTERED_PACKAGES_MAJOR_VERSION */, catalog.packages.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingConstructFrameworkCount" /* MISSING_CONSTRUCT_FRAMEWORK_COUNT */, catalog.packages.filter((pkg) => pkg.constructFramework == null).length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingConstructFrameworkVersionCount" /* MISSING_CONSTRUCT_FRAMEWORK_VERSION_COUNT */, catalog.packages.filter((pkg) => pkg.constructFramework && pkg.constructFramework.majorVersion == null).length, aws_embedded_metrics_1.Unit.Count);
    })();
    // Clean up existing entries if necessary. In particular, remove the license texts as they make
    // the catalog unnecessarily large, and may hinder some search queries' result quality.
    for (const entry of catalog.packages) {
        if (entry.metadata) {
            delete entry.metadata.licenseText;
        }
    }
    // Upload the result to S3 and exit.
    const result = await aws.s3().putObject({
        Bucket: BUCKET_NAME,
        Key: constants.CATALOG_KEY,
        Body: JSON.stringify(catalog, null, 2),
        ContentType: 'application/json',
        CacheControl: 'public, max-age=300, must-revalidate, proxy-revalidate',
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
            'Package-Count': `${catalog.packages.length}`,
        },
    }).promise();
    if (nextStartAfter != null) {
        console.log(`Will continue from ${nextStartAfter} in new invocation...`);
        const nextEvent = { ...event, startAfter: nextStartAfter };
        // We start it asynchronously, as this function has a provisionned
        // concurrency of 1 (so a synchronous attempt would always be throttled).
        await aws.lambda().invokeAsync({
            FunctionName: context.functionName,
            InvokeArgs: JSON.stringify(nextEvent, null, 2),
        }).promise();
    }
    return result;
}
exports.handler = handler;
/**
 * A generator that asynchronously traverses the set of "interesting" objects
 * found by listing the configured S3 bucket. Those objects correspond to all
 * npm package tarballs present under the `packages/` prefix in the bucket.
 *
 * @param bucket the bucket in which to list objects
 * @param startAfter the key to start reading from, if provided.
 */
async function* relevantObjects(bucket, startAfter) {
    var _a, _b;
    const request = {
        Bucket: bucket,
        Prefix: constants.STORAGE_KEY_PREFIX,
        StartAfter: startAfter,
    };
    do {
        const result = await aws.s3().listObjectsV2(request).promise();
        for (const object of (_a = result.Contents) !== null && _a !== void 0 ? _a : []) {
            if (!((_b = object.Key) === null || _b === void 0 ? void 0 : _b.endsWith(constants.PACKAGE_KEY_SUFFIX))) {
                continue;
            }
            // We only register packages if they have AT LEAST docs in one language.
            const tsDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_TYPESCRIPT}`;
            const pyDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_PYTHON}`;
            const javaDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_JAVA}`;
            if (!(await aws.s3ObjectExists(bucket, tsDocs)) &&
                !(await aws.s3ObjectExists(bucket, pyDocs)) &&
                !(await aws.s3ObjectExists(bucket, javaDocs))) {
                continue;
            }
            yield object;
        }
        request.ContinuationToken = result.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
async function appendPackage(packages, pkgKey, bucketName, denyList) {
    var _a, _b, _c;
    console.log(`Processing key: ${pkgKey}`);
    const [, packageName, versionStr] = constants.STORAGE_KEY_FORMAT_REGEX.exec(pkgKey);
    const version = new semver_1.SemVer(versionStr);
    const found = (_a = packages.get(packageName)) === null || _a === void 0 ? void 0 : _a.get(version.major);
    // If the version is === to the current latest, we'll be replacing that (so re-generated metadata are taken into account)
    if (found != null && version.compare(found.version) < 0) {
        console.log(`Skipping ${packageName}@${version} because it is not newer than the existing ${found.version}`);
        return;
    }
    console.log(`Checking if ${packageName}@${version.version} matches a deny list rule`);
    const blocked = denyList.lookup(packageName, version.version);
    if (blocked) {
        console.log(`Skipping ${packageName}@${version.version} because it is blocked by the deny list rule: ${JSON.stringify(blocked)}`);
        return;
    }
    console.log(`Registering ${packageName}@${version}`);
    // Donwload the tarball to inspect the `package.json` data therein.
    const pkg = await aws.s3().getObject({ Bucket: bucketName, Key: pkgKey }).promise();
    const metadataKey = pkgKey.replace(constants.PACKAGE_KEY_SUFFIX, constants.METADATA_KEY_SUFFIX);
    const metadataResponse = await aws.s3().getObject({ Bucket: bucketName, Key: metadataKey }).promise();
    const manifest = await new Promise((ok, ko) => {
        zlib_1.gunzip(Buffer.from(pkg.Body), (err, tar) => {
            if (err) {
                return ko(err);
            }
            tar_stream_1.extract()
                .on('entry', (header, stream, next) => {
                if (header.name !== 'package/package.json') {
                    // Not the file we are looking for, skip ahead (next run-loop tick).
                    return setImmediate(next);
                }
                const chunks = new Array();
                return stream
                    .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                    .once('end', () => {
                    ok(Buffer.concat(chunks));
                    next();
                })
                    .resume();
            })
                .once('finish', () => {
                ko(new Error('Could not find package/package.json in tarball!'));
            })
                .write(tar, (writeErr) => {
                if (writeErr) {
                    ko(writeErr);
                }
            });
        });
    });
    // Add the PackageInfo into the working set
    const pkgMetadata = JSON.parse(manifest.toString('utf-8'));
    const npmMetadata = JSON.parse((_c = (_b = metadataResponse === null || metadataResponse === void 0 ? void 0 : metadataResponse.Body) === null || _b === void 0 ? void 0 : _b.toString('utf-8')) !== null && _c !== void 0 ? _c : '{}');
    const major = new semver_1.SemVer(pkgMetadata.version).major;
    if (!packages.has(pkgMetadata.name)) {
        packages.set(pkgMetadata.name, new Map());
    }
    packages.get(pkgMetadata.name).set(major, {
        author: pkgMetadata.author,
        description: pkgMetadata.description,
        keywords: pkgMetadata.keywords,
        languages: pkgMetadata.jsii.targets,
        license: pkgMetadata.license,
        major,
        metadata: npmMetadata,
        name: pkgMetadata.name,
        version: pkgMetadata.version,
    });
}
//# sourceMappingURL=data:application/json;base64,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