# Copyright (c) 2025 Microsoft Corporation.
# Licensed under the MIT License

"""Cache key creation for Graphrag."""

from typing import Any

from graphrag_llm.cache import create_cache_key

_CACHE_VERSION = 4
"""
If there's a breaking change in what we cache, we should increment this version number to invalidate existing caches.

fnllm was on cache version 2 and though we generate
similar cache keys, the objects stored in cache by fnllm and litellm are different.
Using litellm model providers will not be able to reuse caches generated by fnllm
thus we start with version 3 for litellm.

graphrag-llm package is now on version 4.
This is to account for changes to the ModelConfig that affect the cache key and
occurred when pulling this package out of graphrag.
graphrag-llm, now that is supports metrics, also caches metrics which were not cached before.
"""


def cache_key_creator(
    input_args: dict[str, Any],
) -> str:
    """Generate a cache key based on input arguments.

    Args
    ____
        input_args: dict[str, Any]
            The input arguments for the model call.

    Returns
    -------
        str
            The generated cache key in the format
            `{prefix}_{data_hash}_v{version}` if prefix is provided.
    """
    base_key = create_cache_key(input_args)

    return f"{base_key}_v{_CACHE_VERSION}"
